#pragma once
#include <algorithm>
#include <cstring>
#include <stdexcept>
#include <string>
#include <vector>

namespace quasar {

    class NtpClientException: public std::exception {
    public:
        explicit NtpClientException(std::string inMessage);
        NtpClientException(std::string inMessage, std::string inHost, uint16_t inPort, std::string inIpAddress);

        const char* what() const noexcept override;

        std::string message;
        std::string host;
        uint16_t port;
        std::string ipAddress;

    private:
        mutable std::string whatMessage_;
    };

    class NtpClientExceptionAddrInfo: public NtpClientException {
    public:
        NtpClientExceptionAddrInfo(std::string inHost, uint16_t inPort, int inEaiError, int inErrnum)
            : NtpClientException("getaddrinfo failed: ret=" + std::to_string(inEaiError) + (inErrnum ? ": " + std::to_string(inErrnum) : std::string{}), std::move(inHost), inPort, std::string{})
            , eaiError(inEaiError)
            , errnum(inErrnum)
        {
        }
        int eaiError;
        int errnum;
    };

    class NtpClientExceptionWrongTime: public NtpClientException {
    public:
        NtpClientExceptionWrongTime(std::string inHost, uint16_t inPort, std::string inIpAddress);
    };

    class NtpClientExceptionTimeForgery: public NtpClientException {
    public:
        NtpClientExceptionTimeForgery(std::string inHost, uint16_t inPort, std::string inIpAddress);
    };

    class NtpClientExceptionClockUnsynchronized: public NtpClientException {
    public:
        NtpClientExceptionClockUnsynchronized(std::string inHost, uint16_t inPort, std::string inIpAddress);
    };

    class NtpClientExceptionKissOfDeath: public NtpClientException {
    public:
        static std::string codeToMessage(const std::string& code);

        NtpClientExceptionKissOfDeath(const std::string& inCode, std::string inHost, uint16_t inPort, std::string inIpAddress);
    };

    class NtpClientExceptionDiscoveryFail: public NtpClientException {
    public:
        enum class ErrorId {
            GETADDRINFO,
            SEND,
            TIMEOUT,
            RECV,

            // ntp
            NTP_TIME_FORGERY,
            NTP_UNSYNCHRONIZED,

            // kiss-of-death
            KOD_ACST,
            KOD_AUTH,
            KOD_AUTO,
            KOD_BCST,
            KOD_CRYP,
            KOD_DENY,
            KOD_DROP,
            KOD_RSTR,
            KOD_INIT,
            KOD_MCST,
            KOD_NKEY,
            KOD_NTSN,
            KOD_RATE,
            KOD_RMOT,
            KOD_STEP,
            KOD_OTHER,

            // other
            OTHER,
        };
        struct Host {
            std::string host;
            uint32_t port;
            std::string ipAddress;
            ErrorId errorId;
            int errNo;
            int errOther;

            Host(std::string inHost, uint32_t inPort, std::string inIpAddress, ErrorId inErrorId, int inErrNo = 0, int inErrOther = 0)
                : host(std::move(inHost))
                , port(inPort)
                , ipAddress(std::move(inIpAddress))
                , errorId(inErrorId)
                , errNo(inErrNo)
                , errOther(inErrOther)
            {
            }
        };
        static ErrorId kodToErrorId(const std::string& code);
        NtpClientExceptionDiscoveryFail(std::vector<Host> hosts);
    };

} // namespace quasar
