#include "time_over_http.h"

#include <yandex_io/libs/base/utils.h>

#include <build/scripts/c_templates/svnversion.h>

#include <curl/curl.h>
#include <curl/easy.h>

#include <util/generic/scope.h>

#include <atomic>
#include <cstdlib>
#include <memory>

namespace quasar {
    namespace {
        const char* X_START_TIME = "X-Start-Time:";
        const size_t X_START_TIME_LEN = strlen(X_START_TIME);

        size_t headerParser(char* buffer, size_t size, size_t nitems, void* userdata)
        {
            const size_t len = size * nitems;
            std::optional<int64_t>& timestampUs = *(reinterpret_cast<std::optional<int64_t>*>(userdata));

            if (timestampUs) {
                return len;
            }

            if (X_START_TIME_LEN < len) {
                if (!strncmp(X_START_TIME, buffer, X_START_TIME_LEN)) {
                    std::string value = trim(std::string{buffer + X_START_TIME_LEN, len - X_START_TIME_LEN});
                    char* end = nullptr;
                    int64_t headerTimestampUs = strtoll(value.c_str(), &end, 10);

                    const int64_t minTimestampUs = std::max<int64_t>(1577836800, GetProgramBuildTimestamp()) * 1000000LL; // 2020-01-01
                    const int64_t maxTimestampUs = minTimestampUs + (20 * 365LL * 24LL * 3600LL * 1000000LL);             // minTimestampNs + 20 years
                    if (minTimestampUs < headerTimestampUs && headerTimestampUs < maxTimestampUs) {
                        timestampUs = headerTimestampUs;
                        return len;
                    }
                }
            }
            return len;
        }

    } // namespace

    CurlPerformException::CurlPerformException(const std::string& url, int curlErrorCode)
        : std::runtime_error("Failed to get time from " + url + " (error code: " + std::to_string(curlErrorCode) + ")")
    {
    }

    NoTimestampHeader::NoTimestampHeader(const std::string& url)
        : std::runtime_error("Failed to get http header with timestamp from " + url)
    {
    }

    std::chrono::system_clock::time_point getUtcTimeOverHttp(const std::string& url, std::chrono::milliseconds timeout)
    {
        static std::atomic<CURLcode> rv = curl_global_init(CURL_GLOBAL_ALL);
        if (rv != CURLE_OK) {
            throw std::runtime_error("curl global init failed (error code: " + std::to_string(rv) + ")");
        }

        std::optional<int64_t> timestampUs;
        CURL* curl = curl_easy_init();
        Y_DEFER {
            if (curl) {
                curl_easy_cleanup(curl);
                curl = nullptr;
            }
        };

        curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
        curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS, static_cast<int>(timeout.count()));
        curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT_MS, static_cast<int>(timeout.count()));
        curl_easy_setopt(curl, CURLOPT_NOBODY, 1L);
        curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, headerParser);
        curl_easy_setopt(curl, CURLOPT_HEADERDATA, &timestampUs);

        CURLcode res = curl_easy_perform(curl);
        if (res != CURLE_OK) {
            throw CurlPerformException(url, res);
        }

        if (!timestampUs) {
            throw NoTimestampHeader(url);
        }

        return std::chrono::system_clock::time_point{std::chrono::microseconds{*timestampUs}};
    }

} // namespace quasar
