#pragma once

#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/libs/threading/wakeup_fd.h>

#include <atomic>
#include <chrono>
#include <memory>
#include <string>
#include <unordered_map>
#include <unordered_set>

namespace quasar {

    class Pinger: public std::enable_shared_from_this<Pinger> {
    public:
        using Clock = std::chrono::steady_clock;
        using TimePoint = Clock::time_point;
        using Duration = Clock::duration;

        enum class EventType {
            PACKET_SENT,
            PACKET_RECEIVED,
            PACKET_RECEIVED_DUPLICATE,
            PACKET_LOST,
        };

        enum class SocketType {
            RAW,
            DGRAM,
        };

        struct Event {
            const EventType type;
            const TimePoint time;
            const uint64_t pingId;
            const uint64_t sequenceNumber;
            const std::string host;
            const Duration elapsed;
            const int errorCode;

            explicit Event(
                EventType type,
                const TimePoint& time,
                uint64_t pingId = 0,
                uint64_t sequenceNumber = 0,
                const std::string& host = "",
                const Duration& elapsed = Duration(0),
                int errorCode = 0)
                : type(type)
                , time(time)
                , pingId(pingId)
                , sequenceNumber(sequenceNumber)
                , host(host)
                , elapsed(elapsed)
                , errorCode(errorCode)
            {
            }
        };

        class Listener {
        public:
            virtual ~Listener() = default;

            virtual void onEvent(const Event& event) = 0;
        };

    public:
        Pinger(
            const Duration& interval,
            const Duration& timeout,
            Listener* listener,
            ICallbackQueue* worker,
            SocketType socketType);

        ~Pinger();

        static std::shared_ptr<Pinger> create(
            const Duration& interval,
            const Duration& timeout,
            Listener* listener,
            ICallbackQueue* worker,
            SocketType socketType);

        uint64_t startPing(
            const std::string& host,
            const Duration& interval = Duration(0),
            const Duration& timeout = Duration(0));

        void stopPing(uint64_t pingId);

    private:
        struct Ping {
            const std::string host;
            const Duration interval;
            const Duration timeout;

            Ping(const std::string& host, const Duration& interval, const Duration& timeout)
                : host(host)
                , interval(interval)
                , timeout(timeout)
            {
            }
        };

        struct Request {
            const uint64_t pingId;
            const TimePoint sendTime;

            uint64_t repliesCount = 0; // count replies to detect duplicates

            Request(const uint64_t pingId, const TimePoint& sendTime)
                : pingId(pingId)
                , sendTime(sendTime)
            {
            }
        };

    private:
        void sendPing(uint64_t pingId);
        void scheduleNextPing(uint64_t pingId, const Duration& interval);

        void finalizePing(uint16_t sequenceNumber);
        void scheduleFinalizePing(uint16_t sequenceNumber, const Duration& timeout);

        void waitICMPPackets();
        void acceptICMPPacket();

        void processPingReply(const TimePoint& recvTime, uint16_t sequenceNumber);
        void scheduleProcessPingReply(const TimePoint& recvTime, uint16_t sequenceNumber);

    private:
        std::weak_ptr<Pinger> weakFromThis();

    private:
        std::unordered_map<uint64_t, Ping> pings_;
        std::atomic<uint64_t> pingId_;

        const Duration interval_;
        const Duration timeout_;

        Listener* listener_;

        uint16_t sequenceNumber_;

        ICallbackQueue* worker_;

        const uint64_t pid_;

        int socket_ = -1;
        std::thread socketListener_;
        std::atomic_bool stopped_;
        WakeupFd wakeupFd_;
        const SocketType socketType_;

        std::unordered_map<uint16_t, Request> requests_;
    };

} // namespace quasar
