#include "pstore_collector.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/filesystem/directory_iterator.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/folder/path.h>

#include <iostream>

using namespace quasar;

YIO_DEFINE_LOG_MODULE("pstore");

namespace {

    constexpr std::string_view GZIP_SUFFIX = ".enc.z";

} // namespace

PstoreCollector::PstoreCollector(
    const std::string& pstorePath,
    std::shared_ptr<YandexIO::IDevice> device)
    : pstorePath_(pstorePath)
    , device_(std::move(device))
{
}

void PstoreCollector::collect() {
    if (pstorePath_.empty()) {
        YIO_LOG_ERROR_EVENT("PstoreCollector.EmptyPstorePath", "pstore path is empty");
        return;
    }

    std::unordered_map<std::string, std::string> pstoreData;

    YIO_LOG_INFO("Collecting logs from " << pstorePath_);

    for (auto it = DirectoryIterator(pstorePath_); it.isValid(); it.next()) {
        if (!it.isFile()) {
            continue;
        }

        auto name = it.currentEntryName();
        const auto path = JoinFsPaths(pstorePath_, name);

        YIO_LOG_INFO("Collecting " << path);

        bool compressed = false;
        if (name.ends_with(GZIP_SUFFIX)) {
            compressed = true;
            name = name.substr(0, name.size() - GZIP_SUFFIX.size());
        }

        try {
            auto content = getFileContent(path);
            if (compressed) {
                content = gzipDecompress(content);
            }
            // XXX size limit?
            pstoreData[name] = content;
        } catch (const std::exception& e) {
            pstoreData[name] = std::string();
            YIO_LOG_ERROR_EVENT("PstoreCollector.Collect.Exception", "Failed to collect: " << e.what());
        }

        remove(path.c_str());
    }

    if (pstoreData.empty()) {
        YIO_LOG_INFO("Nothing to collect");
        return;
    }

    device_->telemetry()->reportKeyValues("pstore", pstoreData);
}
