#include "quasar_client.h"

#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/datacratic/public.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>

#include <yandex_io/protos/quasar_proto.pb.h>

#include <google/protobuf/text_format.h>

#include <library/cpp/getopt/small/last_getopt.h>

#include <chrono>
#include <fstream>
#include <limits>
#include <optional>
#include <thread>

using namespace quasar;
using namespace quasar::proto;

namespace {

    struct QuasarClientConfig {
        std::string host;
        int port{-1};
        std::optional<QuasarMessage> message;
        std::string configFile;
    };

    QuasarClientConfig parseConfig(int argc, char** argv) {
        QuasarClientConfig result;

        std::string serviceName;
        TString messageString;

        {
            NLastGetopt::TOpts opts;
            opts.AddLongOption('s', "service")
                .Help("Service we have to connect to")
                .Handler1T<std::string>([&](const std::string& val) {
                    serviceName = val;
                });
            opts.AddLongOption('c', "config-file")
                .Help("Config file from which we take service port. Makes sense only with service param")
                .DefaultValue<std::string>("/system/vendor/quasar/quasar.cfg")
                .Handler1T<std::string>([&](const std::string& val) {
                    result.configFile = val;
                });
            opts.AddLongOption('m', "message")
                .Help("Message which will be sent to the host")
                .Handler1T<std::string>([&](const std::string& val) {
                    messageString = val;
                });
            opts.AddLongOption('h', "host")
                .Help("Host we have to connect to")
                .Handler1T<std::string>([&](const std::string& val) {
                    result.host = val;
                });
            opts.AddLongOption('p', "port")
                .Help("Port we have to connect to")
                .Handler1T<int>([&](const int& val) {
                    result.port = val;
                });
            opts.SetFreeArgsNum(0);

            NLastGetopt::TOptsParseResult r(&opts, argc, argv);
        }

        if (!serviceName.empty()) {
            if (result.port != -1 || !result.host.empty()) {
                throw std::runtime_error("Please specify EITHER host and port OR service name");
            }

            auto configuration = YandexIO::makeConfiguration(result.configFile);
            result.port = configuration->getServicePort(serviceName);
            result.host = "localhost";
        }

        if (!messageString.empty()) {
            QuasarMessage message;
            google::protobuf::TextFormat::ParseFromString(messageString, &message);
            result.message = std::move(message);
        }

        {
            constexpr int minPort = std::numeric_limits<uint16_t>::min();
            constexpr int maxPort = std::numeric_limits<uint16_t>::max();

            if (result.port < minPort || result.port > maxPort) {
                std::stringstream ss;
                ss << "Port should be in range [" << minPort << ", " << maxPort << "], but it's " << result.port << std::endl;
                throw std::runtime_error(ss.str());
            }
        }

        return result;
    }

    void printMessage(const quasar::ipc::SharedMessage& message) {
        TString textFormat;
        google::protobuf::TextFormat::PrintToString(*message, &textFormat);

        std::cout << "----------------------Received---------------------" << std::endl;
        std::cout << textFormat;
        std::cout << "---------------------------------------------------" << std::endl;
    }

    std::shared_ptr<ipc::IConnector> createConnector(const std::string& host, uint16_t port,
                                                     std::shared_ptr<YandexIO::Configuration> configuration) {
        auto connector = ipc::datacratic::createIpcConnector("server", configuration);

        connector->setMessageHandler(printMessage);
        connector->setConnectHandler([&]() {
            std::cout << "Connected to " << host << ":" << port << std::endl;
        });

        connector->setConnectionErrorHandler([&](const std::string& error) {
            std::cout << "Connection error: " << error << std::endl;
        });

        connector->connectToTcpHost(host, port);
        connector->waitUntilConnected();

        return connector;
    }

} // namespace

int RunQuasarClient(int argc, char** argv)
{
    TerminateWaiter waiter;

    QuasarClientConfig config;
    try {
        config = parseConfig(argc, argv);
    } catch (const std::runtime_error& e) {
        std::cerr << "Failed parsing arguments: " << e.what() << std::endl;
        return 1;
    }
    const auto configuration = config.configFile.empty() ? YandexIO::makeConfiguration() : YandexIO::makeConfiguration(config.configFile);
    auto connector = createConnector(config.host, (uint16_t)config.port, configuration);
    if (config.message) {
        connector->sendMessage(std::move(*config.message));
    }

    waiter.wait();

    return 0;
}
