#include "factory.h"

#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("rate_limiter");

using namespace quasar;

std::unique_ptr<BucketRateLimiter> createRateLimiter(const Json::Value& params) {
    std::unique_ptr<BucketRateLimiter> newRateLimiter;

    if (tryGetBool(params, "enabled", true)) {
        uint32_t bucketsCount = tryGetUInt32(params, "buckets_count", 3);
        uint32_t limit = tryGetUInt32(params, "events_limit", 30);
        auto filteredEvents = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(
            params, "filtered_events", {"ysk_error_workingthread_block"});
        bool logOverflows = tryGetBool(params, "log_overflows", false);
        auto rotatingBucketsPeriod = std::chrono::milliseconds(tryGetUInt32(params, "bucket_rotating_period_ms", 10000));

        newRateLimiter = std::make_unique<BucketRateLimiter>(bucketsCount, limit, filteredEvents);
        if (logOverflows) {
            newRateLimiter->setOverflowOnBucketRotatingCb([](const BucketRateLimiter::OverflowInfo& overflowInfo) {
                YIO_LOG_ERROR_EVENT("RateLimiter.OnOverflow", "Overflow event \"" << overflowInfo.event << "\", limit is " << overflowInfo.limit << ", events count is " << overflowInfo.eventCount);
            });
        }
        newRateLimiter->setRotatingBucketsPeriod(rotatingBucketsPeriod);
    }

    return newRateLimiter;
}
