#pragma once

#include "ring_buffer.h"

#include <algorithm>
#include <stdexcept>

namespace quasar {

    template <typename T>
    RingBuffer<T>::RingBuffer(size_t capacity)
        : capacity(capacity)
        , begin(0)
        , end(0)
        , empty(true)
    {
        if (capacity == 0) {
            throw std::invalid_argument("RingBuffer creation with zero capacity is not permitted");
        }
        buffer = new T[capacity];
    }

    template <typename T>
    RingBuffer<T>::~RingBuffer() {
        delete[] buffer;
    }

    template <typename T>
    size_t RingBuffer<T>::write(const T* data, size_t length) {
        if (length == 0) {
            return length;
        }

        if (length >= capacity) {
            std::copy(data + length - capacity, data + length, buffer);
            begin = 0;
            end = 0;
            empty = false;
            return capacity;
        } else {
            if (length > getFree()) {
                begin = (begin + length - getFree()) % capacity;
            }

            size_t updateSize = length;
            size_t tailChunk = std::min(length, capacity - end);
            std::copy(data, data + tailChunk, buffer + end);
            end = (end + tailChunk) % capacity;

            if (tailChunk < updateSize) {
                size_t headChunk = length - tailChunk;
                std::copy(data + tailChunk, data + tailChunk + headChunk, buffer);
                end = headChunk;
            }

            empty = false;
            return length;
        }
    }

    template <typename T>
    size_t RingBuffer<T>::read(T* target, size_t length) const {
        if (length > getSize()) {
            length = getSize();
        }

        if (length == 0) {
            return length;
        }

        size_t tailChunk = std::min(length, capacity - begin);
        std::copy(buffer + begin, buffer + begin + tailChunk, target);
        if (tailChunk < length) {
            size_t headChunk = length - tailChunk;
            std::copy(buffer, buffer + headChunk, target + tailChunk);
        }

        return length;
    }

    template <typename T>
    size_t RingBuffer<T>::getSize() const {
        if ((begin == end) && !empty) {
            return capacity;
        }

        if (begin <= end) {
            return end - begin;
        } else {
            return (capacity - begin) + end;
        }
    }

    template <typename T>
    size_t RingBuffer<T>::getCapacity() const {
        return capacity;
    }

    template <typename T>
    size_t RingBuffer<T>::getFree() const {
        return capacity - getSize();
    }

    template <class T>
    void RingBuffer<T>::clear() {
        begin = end = 0;
        empty = true;
    }

} // namespace quasar
