#include "memory_counter_self_destroyer_predicate.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <sstream>

#include <unistd.h>

YIO_DEFINE_LOG_MODULE("self_destroyer");

using namespace quasar;

MemoryCounterSelfDestroyerPredicate::MemoryCounterSelfDestroyerPredicate(size_t limitKb, std::shared_ptr<MemoryCounter> memoryCounter,
                                                                         bool reportDifference,
                                                                         std::shared_ptr<YandexIO::IDevice> device,
                                                                         std::string processName)
    : limitKB_(limitKb)
    , memoryCounter_(std::move(memoryCounter))
    , reportDifference_(reportDifference)
    , device_(std::move(device))
    , processName_(std::move(processName))
    , pid_(::getpid())
{
}

SelfDestroyerPredicate::Result MemoryCounterSelfDestroyerPredicate::checkNeedDestroySelf() const {
    const ssize_t currentMemory = memoryCounter_->count(pid_);
    if (currentMemory == -1) {
        YIO_LOG_ERROR_EVENT("SelfDestroyerPredicate.CountMemoryFail", "Could not read current memory value for process " << processName_);
        return SelfDestroyerPredicate::Result{false, ""};
    }

    if (reportDifference_) {
        Json::Value freeSize;
        freeSize["sizeKb"] = limitKB_ - currentMemory;
        freeSize["process"] = processName_;
        device_->telemetry()->reportEvent(memoryCounter_->getMetricName(), jsonToString(freeSize));
    }

    if (currentMemory <= limitKB_) {
        return SelfDestroyerPredicate::Result{false, ""};
    }
    std::stringstream reasonStream;
    reasonStream << "Process " << memoryCounter_->getMetricName() << " memory (" << currentMemory << "KB) exceed limit (" << limitKB_ << "KB)";
    return SelfDestroyerPredicate::Result{true, reasonStream.str()};
}
