#include "audio_stream.h"

#include <yandex_io/libs/logging/logging.h>

#include <vector>

using namespace quasar;

bool AudioStream::read(std::vector<std::int16_t>& buffer, size_t size)
{
    std::unique_lock<std::mutex> lock(mutex_);
    buffer.clear();
    while (queue_.size() < size && !stopped_)
    {
        YIO_LOG_TRACE("not enough data: " << queue_.size() << " out of" << size);
        cond_.wait_for(lock, std::chrono::milliseconds(1000));
    }
    if (!stopped_)
    {
        for (size_t i = 0; i < size; ++i)
        {
            auto v = queue_.front();
            queue_.pop();
            buffer.push_back(v);
        }
    }
    YIO_LOG_TRACE("AudioStream(READ) queue size: " << queue_.size());
    return buffer.size() == size;
}

void AudioStream::write(const std::vector<std::int16_t>& buffer)
{
    std::unique_lock<std::mutex> lock(mutex_);
    for (auto item : buffer)
    {
        queue_.push(item);
    }
    cond_.notify_one();
    YIO_LOG_TRACE("AudioStream(WRITE) queue size: " << queue_.size());
}

AudioStream::AudioStream()
{
    std::unique_lock<std::mutex> lock(mutex_);
    stopped_ = false;
}

AudioStream::~AudioStream()
{
    std::unique_lock<std::mutex> lock(mutex_);
    stopped_ = true;
    cond_.notify_all();
}

void AudioStream::start()
{
    std::unique_lock<std::mutex> lock(mutex_);
    stopped_ = false;
}

void AudioStream::stop()
{
    std::unique_lock<std::mutex> lock(mutex_);
    stopped_ = true;
}

void AudioStream::clear()
{
    std::unique_lock<std::mutex> lock(mutex_);
    decltype(queue_) empty;
    queue_.swap(empty);
}
