#pragma once

#include "audio_stream.h"

#include <yandex_io/libs/device/device.h>

#include <atomic>
#include <functional>
#include <string>
#include <vector>

/**
 *
 * <p>Port to C++ of Akimov Sergey's implementation.</p>
 *
 * @author Sergey Akimov (akiserg@yandex-team.ru)
 * @author katayad (katayad@yandex-team.ru)
 */

namespace quasar {

    class SoundDataReceiver {
    public:
        using byte = unsigned char;
        const size_t MAX_WAIT_SECONDS = 20;
        const size_t SPLIT_CHUNKS = 2;
        int frequencyTolerance_;
        int amplitudeThreshold_;

        std::vector<std::vector<double>> split(const std::vector<double>& packet) const;

        std::function<void(const std::vector<unsigned char>& payload, int version)> onDataReceived;
        std::function<void(int)> onUnsupportedProtocol;
        std::function<void()> onTransferStart;
        std::function<void()> onTransferError;

        SoundDataReceiver(std::shared_ptr<YandexIO::IDevice> device, size_t sampleRate);

        ~SoundDataReceiver();

        void start();

        void write(const std::vector<std::int16_t>& buffer);

        void stop();

        void listen();

        bool isHandshakeStartMatching(const std::vector<double>& handshakeStart) const;

        bool findInHandshake(const std::vector<double>& handshakeStart, int handshakeFrequency, int start, int end) const;

        void updateStartHandshake(std::vector<double>& handshakeStart, double dominant) const;

    private:
        static std::vector<byte> decodeBitChunks(int bits, const std::vector<byte>& chunks);
        static std::vector<byte> extractPacket(const std::vector<double>& packet);

        std::shared_ptr<YandexIO::IDevice> device_;
        AudioStream audioStream_;
        const size_t sampleRate_;
        const size_t frameSize_;
        const size_t maxPacketsCount_;
        std::thread listenThread_;
        std::atomic<bool> stopped_{true};
        int handshakeStartMin_;
        int handshakeStartMax_;

        std::vector<byte> getPayload(const std::vector<double>& packet) const;

        void onSoundMessageEnd(std::vector<double>& packet, std::vector<double>& handshakeStart,
                               bool& record, bool& handshakeStartOccurred);
        void onNewMessageFrequency(double dominantFrequency, std::vector<double>& packet, std::vector<double>& handshakeStart,
                                   bool& record, bool& handshakeStartOccurred) const;
        void onNewFrequency(double dominantFrequency, double dominantAmplitude, std::vector<double>& packet, std::vector<double>& handshakeStart,
                            bool& record, bool& handshakeStartOccurred) const;
    };

} // namespace quasar
