#pragma once

#include <yandex_io/libs/error_correcting_code/galois_field.h>
#include <yandex_io/libs/logging/logging.h>

#include <algorithm>
#include <cmath>
#include <cstdint>
#include <complex>
#include <functional>
#include <string>
#include <vector>

/**
 *
 * <p>Most functions are ported to C++ from Akimov Sergey's implementation.</p>
 * <p> Parameters must be same as in
 * https://bitbucket.browser.yandex-team.ru/projects/ML/repos/quasar/browse/android/sounddatatransfer/src/main/java/ru/yandex/quasar/sounddatatransfer/utils/SoundUtils.java </p>
 *
 * @author Sergey Akimov (akiserg@yandex-team.ru)
 * @author katayad (katayad@yandex-team.ru)
 */

namespace quasar {
    namespace SoundUtils {

        const int PROTOCOL_VERSION = 1; // must be less then 32 (we have 5 bits for it)
        const int DEFAULT_SAMPLE_RATE = 16000;

        const double DURATION = 0.04;
        const int FEC_BYTES = 4;      // must be less then 16
        const int CHECKSUM_BYTES = 2; // Must be <= 4 and >= 1

        const int HANDSHAKE_START_HZ = 6192;
        const int HANDSHAKE_END_HZ = 5120 + 512;

        const size_t HANDSHAKE_START_LEN = 4;
        const int HANDSHAKE_START[HANDSHAKE_START_LEN] = {
            HANDSHAKE_START_HZ, HANDSHAKE_START_HZ,
            5120, 5120};
        const size_t HANDSHAKE_END_LEN = 8;
        const int HANDSHAKE_END[HANDSHAKE_END_LEN] = {
            HANDSHAKE_END_HZ, HANDSHAKE_END_HZ,
            HANDSHAKE_END_HZ, HANDSHAKE_END_HZ,
            HANDSHAKE_END_HZ, HANDSHAKE_END_HZ,
            HANDSHAKE_END_HZ, HANDSHAKE_END_HZ};

        const int START_HZ = 1024;
        const int STEP_HZ = 256;
        const int BITS = 4;

        const int TOLERANCE = 100;

        bool isWifiTypeValid(int wifiType);

        /*
         * GaloisField(0x13, 16, 1):
         * GF(2^m) with m-degree primitive
         * Since we are using 4 bits m = 4.
         */

        // x^4 + x + 1
        const GaloisField GALOIS_FIELD = GaloisField(0x13, 16, 1);

        // MAX_BYTES_PER_CHUNK = 2^m - 1
        const int MAX_BYTES_PER_CHUNK = 15;

        bool match(double frequency1, double frequency2, int tolerance);

        int getCheckSum(const std::vector<unsigned char>& payload, int offset, size_t length);

        std::vector<unsigned char> addChecksum(const std::vector<unsigned char>& payload);

        bool isChecksumCorrect(const std::vector<unsigned char>& decoded);

        bool isPartOfStartHandshake(double frequency, int tolerance);

        std::vector<double> filter(const std::vector<double>& frequencies, int tolerance);

        int getProtocolVersionFromFrequencies(const std::vector<double>& frequencies, int chunks, int tolerance);

        std::vector<std::complex<double>> toComplex(const std::vector<std::int16_t>& chunk, int size);

        struct FrequencyRange {
            int min;
            int max;
        };

        int argmax(const std::vector<std::complex<double>>& c, const std::vector<FrequencyRange>& frequencyRanges,
                   const std::vector<double>& freqs, int sampleRate);

        /**
         *
         * <p> created by e-maxx aka maxdiver</p>
         *
         * @author maxdiver
         */

        void fft(std::vector<std::complex<double>>& a, bool invert = false);

        std::vector<double> fftfreq(int n);

        std::pair<double, double> findDominant(const std::vector<std::int16_t>& chunk, int size,
                                               int sampleRate, const std::vector<FrequencyRange>& frequencyRanges);

        std::vector<unsigned char> encodePayload(const std::vector<unsigned char>& payload);

        std::vector<int> getProtocolVersionAsFrequencies(int protocolVersion);

        class BitStreamToneGenerator {
        private:
            const std::vector<int> versionFrequencies;
            const std::vector<unsigned char> encodedPayload;

            size_t startHandshakeCount = 0;
            size_t endHandshakeCount = 0;
            size_t versionCount = 0;
            size_t payloadCount = 0;

        public:
            explicit BitStreamToneGenerator(std::vector<unsigned char> payload, int protocolVersion);

            bool hasNext();
            int next();
            size_t size();
        };

        std::vector<std::int16_t> samplesGeneratorGenerateSamples(int frequency, int sampleRate);

        std::vector<std::vector<std::int16_t>> samplesGeneratorGenerate(BitStreamToneGenerator frequencies, int sampleRate);

        std::vector<std::vector<std::int16_t>> soundDataSourceGenerateSamples(int protocolVersion, std::vector<unsigned char> payload, int sampleRate);

        std::vector<unsigned char> convert(const std::vector<std::int16_t>& sample);
        std::vector<std::int16_t> convert(const std::vector<unsigned char>& data);

    } // namespace SoundUtils

} // namespace quasar
