#pragma once
#include "i_live_data.h"
#include "signal_with_state.h"

#include <atomic>
#include <type_traits>

namespace quasar {

    template <class Value>
    class LiveData: public SignalWithState<typename std::conditional<std::is_base_of<ILiveDataTag, Value>::value, Value, ILiveData<Value>>::type> {
    public:
        using ValueType = typename std::conditional<std::is_base_of<ILiveDataTag, Value>::value, Value, ILiveData<Value>>::type::ValueType;

        LiveData()
        {
            setValue(ValueType{});
        }

        LiveData(ValueType initialValue)
        {
            setValue(std::move(initialValue));
        }

        LiveData(std::nullptr_t) {
        }

        void setValue(ValueType value) {
            (*this)(std::move(value));
        }

        bool try_lock()
        {
            lock_.fetch_add(1);
            return true;
        }

        void lock()
        {
            lock_.fetch_add(1);
        }

        void unlock()
        {
            if (lock_.fetch_sub(1) == 1) {
                bool expected{true};
                if (delayed_.compare_exchange_strong(expected, false)) {
                    SignalWithState<typename std::conditional<std::is_base_of<ILiveDataTag, Value>::value, Value, ILiveData<Value>>::type>::emit();
                }
            }
        }

        LiveData& operator=(const ValueType& value)
        {
            (*this)(value);
            return *this;
        }

        LiveData& operator=(ValueType&& value)
        {
            (*this)(std::move(value));
            return *this;
        }

    public: // ILiveData
        ValueType value() const noexcept override {
            ValueType result;
            SignalWithState<ILiveData<ValueType>>::loadState([&](const ValueType& v) { result = v; });
            return result;
        }

    protected:
        void emit() override {
            if (lock_.load()) {
                delayed_.store(true);
            } else {
                SignalWithState<typename std::conditional<std::is_base_of<ILiveDataTag, Value>::value, Value, ILiveData<Value>>::type>::emit();
            }
        }

    private:
        std::atomic<uint16_t> lock_{0};
        std::atomic<bool> delayed_{false};
    };

} // namespace quasar
