#pragma once

#include "i_signal.h"

#include <yandex_io/libs/threading/i_callback_queue.h>

#include <atomic>
#include <mutex>
#include <type_traits>
#include <utility>
#include <vector>

namespace quasar {

    class SignalPrivate {
    protected:
        static uint32_t allocateConnectionId();
        static void logException();
    };

    template <class S>
    class SignalBase: public S, public SignalPrivate {
    public: // ISignal
        SignalConnectionId connect(typename S::Slot slot, const Lifetime::Tracker& tracker) noexcept override {
            if (!slot) {
                return SignalConnectionId{};
            }
            return implConnect(std::move(slot), tracker, nullptr);
        }

        SignalConnectionId connect(typename S::Slot slot, const Lifetime::Tracker& tracker, std::shared_ptr<ICallbackQueue> callbackQueue) noexcept override {
            if (!callbackQueue) {
                return SignalConnectionId{};
            }
            if (!slot) {
                return SignalConnectionId{};
            }
            return implConnect(std::move(slot), tracker, std::move(callbackQueue));
        }

        bool disconnect(SignalConnectionId connectionId) noexcept override {
            if (!connectionId.value) {
                return false;
            }

            bool result = false;
            {
                std::unique_lock lock(mutex_);
                for (auto& con : *connections_) {
                    if (con->connectionId == connectionId.value) {
                        con->connectionId = 0;
                        result = true;
                    }
                }
            }
            if (result) {
                cleanExpired(1);
            }
            return result;
        }

    protected:
        static_assert(std::has_virtual_destructor<S>::value);
        SignalBase() = default;
        SignalBase(const SignalBase&) = delete;
        SignalBase& operator=(const SignalBase&) = delete;

        std::unique_lock<std::mutex> mutexLock() const {
            return std::unique_lock<std::mutex>(mutex_);
        }

        void forEachSlot(const std::function<void(typename S::Slot)>& slotInvoker) noexcept {
            decltype(connections_) connections;
            size_t dirty = 0;
            {
                std::unique_lock lock(mutex_);
                if (!connections_ || connections_->empty()) {
                    return;
                }
                connections = connections_;
            }

            for (auto& connection : *connections) {
                if (connection->connectionId) {
                    if (auto lock = connection->tracker.lock()) {
                        try {
                            slotInvoker(connection->slot);
                        } catch (...) {
                            logException();
                        }
                    } else {
                        ++dirty;
                    }
                } else {
                    ++dirty;
                }
            }

            if (dirty) {
                cleanExpired(dirty);
            }
        }

    private:
        void cleanExpired(size_t dirtyExpected = 0)
        {
            auto newConnections = std::make_shared<std::vector<ConnectionPtr>>();
            {
                std::unique_lock lock(mutex_);
                if (!connections_) {
                    return;
                }
                newConnections->reserve(connections_->size() < dirtyExpected ? 0 : connections_->size() - dirtyExpected);
                for (auto& con : *connections_) {
                    if (con->connectionId && con->tracker.lock()) {
                        newConnections->emplace_back(con);
                    }
                }
                connections_.swap(newConnections);
            }
            newConnections.reset(); //  reset expired connection out of mutex
        }

        virtual void beforeConnect(const typename S::Slot& /* slot */, const Lifetime::Tracker& /* tracker */, const std::shared_ptr<ICallbackQueue>& /* callbackQueue */) noexcept {
        }

        SignalConnectionId implConnect(typename S::Slot slot, const Lifetime::Tracker& tracker, std::shared_ptr<ICallbackQueue> callbackQueue) noexcept {
            beforeConnect(slot, tracker, callbackQueue);
            std::unique_lock lock(mutex_);
            uint32_t connectionId = allocateConnectionId();
            if (!connections_) {
                connections_ = std::make_shared<std::vector<ConnectionPtr>>();
            } else if (!connections_.unique()) {
                auto newConnections = std::make_shared<std::vector<ConnectionPtr>>();
                newConnections->reserve(connections_->size() + 1);
                newConnections->assign(connections_->begin(), connections_->end());
                connections_ = newConnections;
            }
            if (callbackQueue) {
                std::weak_ptr<ICallbackQueue> callbackQueueWeak = callbackQueue;
                connections_->emplace_back(std::make_shared<Connection>(Connection{
                    connectionId,
                    [tracker, slot{std::move(slot)}, callbackQueueWeak](auto... args)
                    {
                        if (auto cq = callbackQueueWeak.lock()) {
                            cq->add(
                                [tracker, slot, args = std::make_tuple(std::forward<decltype(args)>(args)...)]() mutable {
                                    if (auto lock = tracker.lock()) {
                                        std::apply(std::move(slot), std::move(args));
                                    }
                                });
                        }
                    }, tracker}));
            } else {
                connections_->emplace_back(std::make_shared<Connection>(connectionId, std::move(slot), tracker));
            }
            return SignalConnectionId{connectionId};
        }

    private:
        struct Connection {
            uint32_t connectionId;
            typename S::Slot slot;
            Lifetime::Tracker tracker;
            Connection(uint32_t id, typename S::Slot s, Lifetime::Tracker t)
                : connectionId(id)
                , slot(std::move(s))
                , tracker(std::move(t))
            {
            }
        };
        using ConnectionPtr = std::shared_ptr<Connection>;

        mutable std::mutex mutex_;
        std::shared_ptr<std::vector<ConnectionPtr>> connections_;
    };

} // namespace quasar
