#include "signal_ready.h"

#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

YIO_DEFINE_LOG_MODULE("signal");

namespace {
    uint32_t allocateConnectionId()
    {
        static std::atomic<uint32_t> id{std::numeric_limits<uint32_t>::max() / 2};
        return ++id;
    }
    void invoke(std::function<void()>& slot) noexcept {
        try {
            auto xSlot(std::move(slot));
            slot = nullptr;
            xSlot();
        } catch (const std::exception& ex) {
            YIO_LOG_ERROR_EVENT("Signal.SlotException.LogException", "Unexpected exception in signal slot: " << ex.what());
        } catch (...) {
            YIO_LOG_ERROR_EVENT("Signal.SlotException.LogUnknownError", "Unexpected exception in signal slot: ...");
        }
    }
} // namespace

bool SignalReady::isReady() const noexcept {
    std::unique_lock lock(mutex_);
    return fReady_;
}

void SignalReady::operator()() noexcept {
    decltype(slots_) slots;
    std::unique_lock lock(mutex_);
    if (!fReady_) {
        fReady_ = true;
        slots = std::move(slots_);
    }
    lock.unlock();
    for (auto& [_, slot] : slots) {
        invoke(slot);
    }
}

SignalConnectionId SignalReady::connect(Slot slot, const Lifetime::Tracker& tracker) noexcept {
    if (!slot) {
        return SignalConnectionId{};
    }

    std::unique_lock lock(mutex_);
    if (!fReady_) {
        uint32_t connectionId = allocateConnectionId();
        slots_[connectionId] = makeSafeCallback(std::move(slot), tracker);
        return SignalConnectionId{connectionId};
    }
    lock.unlock();
    if (auto lock = tracker.lock()) {
        slot();
    }
    return SignalConnectionId{};
}

SignalConnectionId SignalReady::connect(Slot slot, const Lifetime::Tracker& tracker, std::shared_ptr<ICallbackQueue> callbackQueue) noexcept {
    if (!slot || !callbackQueue) {
        return SignalConnectionId{};
    }

    std::unique_lock lock(mutex_);
    if (!fReady_) {
        std::weak_ptr<ICallbackQueue> wCallbackQueue = callbackQueue;
        auto qSlot = [slot{std::move(slot)}, tracker, wCallbackQueue{std::move(wCallbackQueue)}]() mutable {
            if (auto cq = wCallbackQueue.lock()) {
                cq->add(std::move(slot), tracker);
            }
        };
        uint32_t connectionId = allocateConnectionId();
        slots_[connectionId] = std::move(qSlot);
        return SignalConnectionId{connectionId};
    }
    lock.unlock();
    callbackQueue->add(std::move(slot), tracker);
    return SignalConnectionId{};
}

bool SignalReady::disconnect(SignalConnectionId signalConnectionId) noexcept {
    std::unique_lock lock(mutex_);
    return slots_.erase(signalConnectionId.value) > 0;
}
