#pragma once
#include "signal_base.h"

namespace quasar {

    template <class S>
    class SignalWithState: public SignalBase<S> {
    public:
        SignalWithState() = default;
        SignalWithState(const SignalWithState&) = delete;
        SignalWithState& operator=(const SignalWithState&) = delete;

        template <typename... Args>
        void operator()(Args... args) noexcept {
            {
                auto lock = SignalBase<S>::mutexLock();
                state_ =
                    [argsTuple = std::make_tuple(std::forward<decltype(args)>(args)...)](const typename S::Slot& slot)
                {
                    std::apply(slot, argsTuple);
                };
            }
            emit();
        }

        bool loadState(const typename S::Slot& slot) const {
            auto lock = SignalBase<S>::mutexLock();
            if (state_) {
                state_(slot);
                return true;
            }
            return false;
        }

    protected:
        virtual void emit()
        {
            if (!state_) {
                return;
            }

            SignalBase<S>::forEachSlot(
                [&](const typename S::Slot& slot)
                {
                    state_(slot);
                });
        }

    private:
        void beforeConnect(const typename S::Slot& slot, const Lifetime::Tracker& tracker, const std::shared_ptr<ICallbackQueue>& callbackQueue) noexcept override {
            if (!callbackQueue) {
                if (auto lock = tracker.lock()) {
                    decltype(state_) state;
                    {
                        auto lock = SignalBase<S>::mutexLock();
                        if (!state_) {
                            return;
                        }
                        state = state_;
                    }
                    state(slot);
                }
            } else {
                auto lock = SignalBase<S>::mutexLock();
                if (!state_) {
                    return;
                }
                callbackQueue->add(
                    [slot, state = state_, tracker] {
                        if (auto lock = tracker.lock()) {
                            state(slot);
                        }
                    });
            }
        }

    private:
        std::function<void(const typename S::Slot& slot)> state_;
    };

} // namespace quasar
