#include "speechkit_logger.h"

#include <yandex_io/libs/logging/logging.h>

#include <contrib/restricted/spdlog/include/spdlog/spdlog.h>

using namespace YandexIO;

namespace {

    const std::unordered_map<std::string, SpeechKit::Logger::LogLevel> logLevelMap = {
        {"fatal", SpeechKit::Logger::LOG_ERROR},
        {"error", SpeechKit::Logger::LOG_ERROR},
        {"warn", SpeechKit::Logger::LOG_WARN},
        {"info", SpeechKit::Logger::LOG_INFO},
        {"debug", SpeechKit::Logger::LOG_DEBUG},
        {"trace", SpeechKit::Logger::LOG_DEBUG},
    };

} // namespace

void YandexIO::setSpeechkitLogLevel(const std::string& logLevelStr) {
    const auto logLevel = logLevelMap.at(logLevelStr);

    auto skLogger = SpeechKit::Logger::getInstance();
    if (skLogger->getLogLevel() != logLevel) {
        skLogger->setLogLevel(logLevel);
    }
}

void SpeechkitLogger::log(const Message& message) const {
    auto level = ::spdlog::level::trace;
    switch (message.level) {
        case ::SpeechKit::Logger::LOG_DEBUG:
            level = ::spdlog::level::debug;
            break;
        case ::SpeechKit::Logger::LOG_INFO:
            level = ::spdlog::level::info;
            break;
        case ::SpeechKit::Logger::LOG_WARN:
            level = ::spdlog::level::warn;
            break;
        case ::SpeechKit::Logger::LOG_ERROR:
            level = ::spdlog::level::err;
            break;
        default:
            YIO_LOG_WARN("Unknown speechkit logLevel for message " << message.text << ": " << message.level);
            return;
    }

    if (auto logger = ::spdlog::default_logger(); logger && logger->should_log(level)) {
        auto sourceLoc = ::spdlog::source_loc{
            message.sourceFilename,
            message.sourceLine,
            message.sourceFunction,
        };
        auto logLine = formatLogLine(message, LF_SOURCE_FUNCTION | LF_THIS_OBJECT | LF_TEXT);
        if (level == ::spdlog::level::err) {
            // as if YIO_LOG_ERROR_EVENT
            std::ostringstream fullLogLine;
            if (message.eventName) {
                fullLogLine << YIO_BUILD_LOG_EVENT_HEADER_IMPL(<<, "speechkit", message.eventName);
            } else {
                fullLogLine << YIO_BUILD_LOG_EVENT_HEADER_IMPL("", "speechkit", "Speechkit.LogError");
            }
            fullLogLine << logLine;
            logger->log(sourceLoc, level, fullLogLine.str());
        } else {
            // as if YIO_LOG_WARN, etc...
            logger->log(sourceLoc, level, logLine);
        }
    }
}
