#include "sqlite_database.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/generic/scope.h>

#include <stdexcept>

#include <sys/stat.h>

YIO_DEFINE_LOG_MODULE("sqlite");

using quasar::SqliteDatabase;

SqliteDatabase::SqliteDatabase(std::string databasePath, uint64_t maxSizeKB /* = -1 */)
    : dbFilename_(std::move(databasePath))
    , maxDbSizeKb_(maxSizeKB)
{
    initDatabase();
}

void SqliteDatabase::initDatabase() {
    sqlite3_config(SQLITE_CONFIG_MULTITHREAD);
    int resCode = sqlite3_open(dbFilename_.c_str(), &db_);
    if (resCode != SQLITE_OK) {
        auto message = std::string("Error when opening database at {") + dbFilename_ + "} : " + sqlite3_errmsg(db_);
        sqlite3_close(db_);
        throw std::runtime_error(message);
    }
}

SqliteDatabase::~SqliteDatabase() {
    cleanup();
}

void SqliteDatabase::cleanup() {
    sqlite3_close(db_);
}

void SqliteDatabase::logQueryError(std::string_view sql, std::string_view message) {
    YIO_LOG_ERROR_EVENT("SqliteDatabase.LogError", "Error occurred when executing query (" << sql << "). Database: " << dbFilename_ << ". Error message: " << message);
}

bool SqliteDatabase::isEnoughMemoryForInsert(const std::string& /* newItem */) {
    /* TODO[labudidabudai]: we check file size before insertion, so after insertion we can exceed our limit on the value of page_size (4096)
     * it is not big problem, but it probably can be solved somehow
     */
    if (maxDbSizeKb_ < 0) {
        return true;
    }

    const int dbSize = getDbSizeUnlocked();
    if (dbSize < 0) {
        return false;
    }
    if (dbSize > maxDbSizeKb_ * 1024) {
        return false;
    }
    return true;
}

void SqliteDatabase::vacuum() {
    runQueryWithoutCallback("VACUUM");
}

void SqliteDatabase::runQueryWithoutCallback(const std::string& query) {
    // TODO[labudidabudai] можно присобачить коллбэк
    char* errmsg = nullptr;
    int resCode = sqlite3_exec(db_, query.c_str(), nullptr, nullptr, &errmsg);
    if (resCode != SQLITE_OK) {
        logQueryError(query, errmsg);
        sqlite3_free(errmsg);
    }
}

int SqliteDatabase::getDatabaseSizeInBytes() {
    std::lock_guard<std::mutex> lock(mutex_);
    return getDbSizeUnlocked();
}

int SqliteDatabase::getDbSizeUnlocked() {
    struct stat status {};
    if (stat(dbFilename_.c_str(), &status)) {
        YIO_LOG_ERROR_EVENT("SqliteDatabase.StatDbFailed", "Cannot get stat from file " << dbFilename_.c_str() << ", error: " << errno << ", " << strError(errno));
        return -errno;
    }
    return status.st_size;
}

const std::string& SqliteDatabase::getDbFilename() const {
    return dbFilename_;
}

void SqliteDatabase::backupToFile(const std::string& backupFilename) {
    TransactionFile backupFile(backupFilename, backupFilename + "_tmp");

    sqlite3* backupDb = nullptr;
    int rc = sqlite3_open(backupFile.temporaryFilename().c_str(), &backupDb);
    Y_DEFER {
        sqlite3_close(backupDb);
    };

    if (rc == SQLITE_OK) {
        std::lock_guard<std::mutex> lock(mutex_);
        sqlite3_backup* backupHandle = sqlite3_backup_init(backupDb, "main", db_, "main");
        if (backupHandle) {
            do {
                rc = sqlite3_backup_step(backupHandle, 5);
                if (rc == SQLITE_OK || rc == SQLITE_BUSY || rc == SQLITE_LOCKED) {
                    sqlite3_sleep(250);
                }
            } while (rc == SQLITE_OK || rc == SQLITE_BUSY || rc == SQLITE_LOCKED);

            sqlite3_backup_finish(backupHandle);
        }
        rc = sqlite3_errcode(backupDb);
    }

    if (rc == SQLITE_OK) {
        backupFile.commit();
    } else {
        auto message = std::string("Error while creating backup for database {" + dbFilename_ + "}: ") + sqlite3_errmsg(backupDb);
        throw std::runtime_error(message);
    }
}

void SqliteDatabase::checkIntegrity() {
    std::lock_guard<std::mutex> lock(mutex_);
    sqlite3_busy_timeout(db_, 5000); // 5 sec

    char* errmsg = nullptr;
    int resCode = sqlite3_exec(db_, "PRAGMA integrity_check", nullptr, nullptr, &errmsg);
    sqlite3_busy_timeout(db_, 0);

    if (resCode != SQLITE_OK) {
        const auto message = std::string("Integrity check failed for database {" + dbFilename_ + "}: ") + errmsg;
        sqlite3_free(errmsg);
        throw std::runtime_error(message);
    }
}
