#include "stop_watch.h"

#include <yandex_io/libs/logging/logging.h>

IStopWatch::~IStopWatch() = default;
ISessionStopWatch::~ISessionStopWatch() = default;

void NullStopWatch::start(const std::string& /*tag*/) {
    // no-op
}

void NullStopWatch::stopAndLog(const std::string& /*tag*/) {
    // no-op
}

void StopWatch::start(const std::string& tag) {
    tags_[tag] = Clock::now();
}

void StopWatch::stopAndLog(const std::string& tag) {
    auto it = tags_.find(tag);
    if (it == tags_.end()) {
        return;
    }

    const auto diff = std::chrono::duration_cast<std::chrono::nanoseconds>(Clock::now() - it->second);
    YIO_LOG_INFO("Time diff for tag: " << tag << ' ' << diff.count() << " ns");

    tags_.erase(it);
}

void LockedStopWatch::start(const std::string& tag) {
    std::scoped_lock guard(mutex_);
    StopWatch::start(tag);
}

void LockedStopWatch::stopAndLog(const std::string& tag) {
    std::scoped_lock guard(mutex_);
    StopWatch::stopAndLog(tag);
}

ISessionStopWatch::SessionID NullSessionStopWatch::startSession() {
    // no-op
    return 0;
}
void NullSessionStopWatch::stopSession(SessionID /*id*/) {
    // no-op
}
void NullSessionStopWatch::start(SessionID /*session*/, const std::string& /*tag*/) {
    // no-op
}

void NullSessionStopWatch::stopAndLog(SessionID /*session*/, const std::string& /*tag*/) {
    // no-op
}

SessionStopWatch::SessionStopWatch(std::chrono::milliseconds minSessionsTimediff)
    : minSessionsTimediff_(minSessionsTimediff)
{
}

ISessionStopWatch::SessionID SessionStopWatch::startSession() {
    const auto now = std::chrono::steady_clock::now();
    // handle first session
    if (!lastSessionTp_.has_value()) {
        lastSessionTp_ = now;
        return nextSession();
    }
    const auto diff = std::chrono::duration_cast<std::chrono::nanoseconds>(now - lastSessionTp_.value());
    if (diff < minSessionsTimediff_) {
        // skip this session
        return NOT_ACTIVE_SESSION;
    }
    // Started new session: log everything in it
    lastSessionTp_ = now;
    return nextSession();
}

ISessionStopWatch::SessionID SessionStopWatch::nextSession() {
    return ++lastSessionId_;
}

void SessionStopWatch::start(SessionID session, const std::string& tag) {
    if (session == NOT_ACTIVE_SESSION) {
        return;
    }
    tags_[session][tag] = Clock::now();
}

void SessionStopWatch::stopAndLog(SessionID session, const std::string& tag) {
    if (session == NOT_ACTIVE_SESSION) {
        return;
    }
    auto sit = tags_.find(session);
    if (sit == tags_.end()) {
        // session not found
        return;
    }
    auto it = sit->second.find(tag);
    if (it == sit->second.end()) {
        // tag in this session not found
        return;
    }

    const auto diff = std::chrono::duration_cast<std::chrono::nanoseconds>(Clock::now() - it->second);
    YIO_LOG_INFO("Session: " << session << ". Time diff for tag: " << tag << ' ' << diff.count() << " ns");

    tags_[session].erase(it);
}

void SessionStopWatch::stopSession(SessionID session) {
    tags_.erase(session);
}

LockedSessionStopWatch::LockedSessionStopWatch(std::chrono::milliseconds minSessionsTimediff)
    : SessionStopWatch(minSessionsTimediff)
{
}

ISessionStopWatch::SessionID LockedSessionStopWatch::startSession() {
    std::scoped_lock guard(mutex_);
    return SessionStopWatch::startSession();
}

void LockedSessionStopWatch::stopSession(SessionID session) {
    std::scoped_lock guard(mutex_);
    SessionStopWatch::stopSession(session);
}

void LockedSessionStopWatch::start(SessionID session, const std::string& tag) {
    std::scoped_lock guard(mutex_);
    SessionStopWatch::start(session, tag);
}

void LockedSessionStopWatch::stopAndLog(SessionID session, const std::string& tag) {
    std::scoped_lock guard(mutex_);
    SessionStopWatch::stopAndLog(session, tag);
}

SessionStatsStopWatch::Stats::Stats(std::chrono::nanoseconds diff)
    : min(diff)
    , max(diff)
    , avg(diff)
    , count(1)
{
}

SessionStatsStopWatch::SessionStatsStopWatch(SessionStatsCallback sessionStatsCallback, std::chrono::milliseconds minStatsInterval)
    : sessionStatsCallback_(std::move(sessionStatsCallback))
    , minStatsInterval_(minStatsInterval)
{
}

ISessionStopWatch::SessionID SessionStatsStopWatch::startSession() {
    if (!statsCollectionStartTime_.has_value()) {
        statsCollectionStartTime_ = Clock::now();
    }
    return nextSession();
}

void SessionStatsStopWatch::stopSession(SessionID session) {
    tags_.erase(session);

    auto now = Clock::now();
    if (now - statsCollectionStartTime_.value_or(now) >= minStatsInterval_) {
        if (sessionStatsCallback_) {
            sessionStatsCallback_(tagToStats_);
        }
        tagToStats_.clear();
        statsCollectionStartTime_.reset();
    }
}

void SessionStatsStopWatch::start(SessionID session, const std::string& tag) {
    tags_[session][tag] = Clock::now();
}

void SessionStatsStopWatch::stopAndLog(SessionID session, const std::string& tag) {
    auto sit = tags_.find(session);
    if (sit == tags_.end()) {
        // session not found
        return;
    }
    auto it = sit->second.find(tag);
    if (it == sit->second.end()) {
        // tag in this session not found
        return;
    }

    const auto diff = std::chrono::duration_cast<std::chrono::nanoseconds>(Clock::now() - it->second);

    auto statsIt = tagToStats_.find(tag);
    if (statsIt == tagToStats_.end()) {
        tagToStats_.emplace(tag, Stats{diff});
    } else {
        auto& stats = statsIt->second;
        ++stats.count;

        if (diff < stats.min) {
            stats.min = diff;
        }
        if (diff > stats.max) {
            stats.max = diff;
        }
        stats.avg += (diff - stats.avg) / stats.count;
    }

    tags_[session].erase(it);
}

ISessionStopWatch::SessionID SessionStatsStopWatch::nextSession() {
    return ++lastSessionId_;
}

LockedSessionStatsStopWatch::LockedSessionStatsStopWatch(SessionStatsCallback sessionStatsCallback, std::chrono::milliseconds minStatsInterval)
    : SessionStatsStopWatch(sessionStatsCallback, minStatsInterval)
{
}

ISessionStopWatch::SessionID LockedSessionStatsStopWatch::startSession() {
    std::scoped_lock guard(mutex_);
    return SessionStatsStopWatch::startSession();
}

void LockedSessionStatsStopWatch::stopSession(SessionID session) {
    std::scoped_lock guard(mutex_);
    SessionStatsStopWatch::stopSession(session);
}

void LockedSessionStatsStopWatch::start(SessionID session, const std::string& tag) {
    std::scoped_lock guard(mutex_);
    SessionStatsStopWatch::start(session, tag);
}

void LockedSessionStatsStopWatch::stopAndLog(SessionID session, const std::string& tag) {
    std::scoped_lock guard(mutex_);
    SessionStatsStopWatch::stopAndLog(session, tag);
}
