#pragma once

#include <chrono>
#include <map>
#include <mutex>
#include <optional>
#include <string>

class IStopWatch {
public:
    virtual ~IStopWatch();
    // save tag and current timestamp
    virtual void start(const std::string& tag) = 0;
    // log timediff btw now and saved timestamp, and drop tag
    virtual void stopAndLog(const std::string& tag) = 0;
};

class NullStopWatch: public IStopWatch {
public:
    void start(const std::string& tag) override;
    void stopAndLog(const std::string& tag) override;
};

class StopWatch: public IStopWatch {
public:
    void start(const std::string& tag) override;
    void stopAndLog(const std::string& tag) override;

private:
    using Clock = std::chrono::steady_clock;
    std::map<std::string, Clock::time_point> tags_;
};

class LockedStopWatch: public StopWatch {
public:
    void start(const std::string& tag) override;
    void stopAndLog(const std::string& tag) override;

private:
    std::mutex mutex_;
};

class ISessionStopWatch {
public:
    virtual ~ISessionStopWatch();
    using SessionID = int32_t;
    /* Start new session */
    virtual SessionID startSession() = 0;
    /* Drop all 'started' tags*/
    virtual void stopSession(SessionID) = 0;

    // save tag and current timestamp
    virtual void start(SessionID session, const std::string& tag) = 0;
    // log timediff btw now and saved timestamp, and drop tag
    virtual void stopAndLog(SessionID session, const std::string& tag) = 0;
};

class NullSessionStopWatch: public ISessionStopWatch {
public:
    SessionID startSession() override;
    void stopSession(SessionID session) override;
    void start(SessionID session, const std::string& tag) override;
    void stopAndLog(SessionID session, const std::string& tag) override;
};

class SessionStopWatch: public ISessionStopWatch {
public:
    explicit SessionStopWatch(std::chrono::milliseconds minSessionsTimediff);
    /**
     * @brief Manage "sessions": If timediff btw current and last "active" startSession call will be
     *        less than minSessionsTimediff -> session will not be active and 'start'/'stopAndLog' methods
     *        will skip logging during this session. If timediff will be greater than minSessionsTimediff
     *        than this session will be active and logs will work
     */
    SessionID startSession() override;
    void stopSession(SessionID /*session*/) override;
    void start(SessionID session, const std::string& tag) override;
    void stopAndLog(SessionID session, const std::string& tag) override;

private:
    using Clock = std::chrono::steady_clock;
    using TagsMap = std::map<std::string, Clock::time_point>;
    using SessionToTags = std::map<SessionID, TagsMap>;

    SessionID nextSession();

    static constexpr SessionID NOT_ACTIVE_SESSION = -1;
    SessionID lastSessionId_{0};
    SessionToTags tags_;

    const std::chrono::milliseconds minSessionsTimediff_;
    std::optional<std::chrono::steady_clock::time_point> lastSessionTp_;
};

class LockedSessionStopWatch: public SessionStopWatch {
public:
    explicit LockedSessionStopWatch(std::chrono::milliseconds minSessionsTimediff_);
    SessionID startSession() override;
    void stopSession(SessionID session) override;
    void start(SessionID session, const std::string& tag) override;
    void stopAndLog(SessionID session, const std::string& tag) override;

private:
    std::mutex mutex_;
};

class SessionStatsStopWatch: public ISessionStopWatch {
public:
    struct Stats {
        std::chrono::nanoseconds min;
        std::chrono::nanoseconds max;
        std::chrono::nanoseconds avg;
        uint32_t count;

        explicit Stats(std::chrono::nanoseconds diff);
    };
    using TagToStats = std::map<std::string, Stats>;
    using SessionStatsCallback = std::function<void(const TagToStats&)>;

    SessionStatsStopWatch(SessionStatsCallback sessionStatsCallback, std::chrono::milliseconds minStatsInterval);

    SessionID startSession() override;
    void stopSession(SessionID session) override;
    void start(SessionID session, const std::string& tag) override;
    void stopAndLog(SessionID session, const std::string& tag) override;

private:
    SessionID nextSession();

private:
    const SessionStatsCallback sessionStatsCallback_;
    const std::chrono::milliseconds minStatsInterval_;

    using Clock = std::chrono::steady_clock;
    using TagsMap = std::map<std::string, Clock::time_point>;
    using SessionToTags = std::map<SessionID, TagsMap>;
    SessionToTags tags_;

    SessionID lastSessionId_{0};
    TagToStats tagToStats_;
    std::optional<Clock::time_point> statsCollectionStartTime_;
};

class LockedSessionStatsStopWatch: public SessionStatsStopWatch {
public:
    LockedSessionStatsStopWatch(SessionStatsCallback sessionStatsCallback, std::chrono::milliseconds minStatsInterval);
    SessionID startSession() override;
    void stopSession(SessionID session) override;
    void start(SessionID session, const std::string& tag) override;
    void stopAndLog(SessionID session, const std::string& tag) override;

private:
    std::mutex mutex_;
};
