#include "wav_file.h"

#include <util/system/byteorder.h>

#include <cstdint>
#include <cstring>
#include <string>

namespace {

    struct WAVHeader {
        char chunkID[4];
        int32_t chunkSize;
        char format[4];

        char subChunk1ID[4];
        int32_t subChunk1Size;
        int16_t audioFormat;
        int16_t numChannels;
        int32_t sampleRate;
        int32_t byteRate;
        int16_t blockAlign;
        int16_t bitsPerSample;

        char subChunk2ID[4];
        int32_t subChunk2Size;
    };

} // namespace

namespace SweepToneGen {

    WavFile::WavFile(const SoundInfo& soundInfo)
        : bytesWritten(0)
        , soundInfo(soundInfo)
    {
    }

    WavFile::~WavFile() {
        // specify member owner to disable virtual dispatching
        WavFile::close();
    }

    bool WavFile::init(const std::string& filename) {
        bytesWritten = 0;

        if (!File::init(filename)) {
            return false;
        }

        if (!writeHeader()) {
            File::close();
            return false;
        }

        return true;
    }

    bool WavFile::write(const void* buf, size_t len) {
        if (!File::write(buf, len)) {
            return false;
        }

        bytesWritten += len;

        return true;
    }

    void WavFile::close() {
        patchHeader();
        File::close();
    }

    bool WavFile::writeHeader() {
        struct WAVHeader header;
        std::memcpy(header.chunkID, "RIFF", 4);
        header.chunkSize = 0;
        std::memcpy(header.format, "WAVE", 4);

        std::memcpy(header.subChunk1ID, "fmt ", 4);
        header.subChunk1Size = HostToLittle<int32_t>(16);
        header.audioFormat = HostToLittle<int16_t>(1);
        header.numChannels = HostToLittle<int16_t>(soundInfo.channelCount);
        header.sampleRate = HostToLittle<int32_t>(soundInfo.sampleRate);
        header.byteRate = HostToLittle<int32_t>(soundInfo.sampleRate * soundInfo.channelCount * soundInfo.sampleSize);
        header.blockAlign = HostToLittle<int16_t>(soundInfo.channelCount * soundInfo.sampleSize);
        header.bitsPerSample = HostToLittle<int16_t>(soundInfo.sampleSize * 8);

        std::memcpy(header.subChunk2ID, "data", 4);
        header.subChunk2Size = 0;

        if (!File::write(/* pos = */ 0, &header, sizeof(header))) {
            return false;
        }

        return true;
    }

    void WavFile::patchHeader() {
        const auto val4offset = HostToLittle<uint32_t>(36 + bytesWritten);
        File::write(/* pos = */ 4, &val4offset, sizeof(val4offset));

        const auto val40offset = HostToLittle<uint32_t>(bytesWritten);
        File::write(/* pos = */ 40, &val40offset, sizeof(val40offset));
    }

} // namespace SweepToneGen
