#include "file_telemetry.h"

#include <yandex_io/libs/telemetry/null/null_params.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>

using namespace YandexIO;

FileTelemetry::FileTelemetry(const std::string& outputFilePath)
    : outputFile_(outputFilePath, std::ios_base::out | std::ios_base::app)
    , uuid_(quasar::makeUUID())
{
    outputFile_.exceptions(std::ios_base::badbit);
}

FileTelemetry::~FileTelemetry() {
}

void FileTelemetry::reportEvent(const std::string& event, YandexIO::ITelemetry::Flags flags) {
    reportEvent(event, "{}", flags);
}

void FileTelemetry::reportEvent(const std::string& event, const std::string& eventJson, ITelemetry::Flags /*flags*/) {
    Json::Value value;
    value["type"] = "EVENT";
    value["name"] = event;
    value["value"] = eventJson;
    writeToFile(quasar::jsonToString(value));
}

void FileTelemetry::reportError(const std::string& error, const std::string& errorValue, ITelemetry::Flags /*flags*/) {
    Json::Value value;
    value["type"] = "ERROR";
    value["name"] = error;
    value["value"] = errorValue;
    writeToFile(quasar::jsonToString(value));
}

void FileTelemetry::reportError(const std::string& error, ITelemetry::Flags flags) {
    reportError(error, "{}", flags);
}

void FileTelemetry::reportLogError(const std::string& message, const std::string& sourceFileName, size_t sourceLine, const std::string& eventJson)
{
    Json::Value value;

    if (!eventJson.empty()) {
        auto maybeEventJsonParsed = quasar::tryParseJson(eventJson);
        if (maybeEventJsonParsed) {
            value = std::move(*maybeEventJsonParsed);
        } else {
            // Cannot use error logging: may induce new LogError events
            value["bad_event"] = eventJson;
        }
    }

    value["type"] = "YIO_LOG_ERROR";
    value["log"] = message;
    value["file_name"] = sourceFileName;
    value["file_line"] = sourceLine;
    writeToFile(quasar::jsonToString(value));
}

void FileTelemetry::reportKeyValues(const std::string& eventName, const std::unordered_map<std::string, std::string>& keyValues, ITelemetry::Flags flags) {
    Json::Value eventValue;
    for (const auto& keyValue : keyValues)
    {
        eventValue[keyValue.first] = keyValue.second;
    }

    reportEvent(eventName, quasar::jsonToString(eventValue), flags);
}

void FileTelemetry::putAppEnvironmentValue(const std::string& key, const std::string& value) {
    {
        std::scoped_lock<std::mutex> scopedLock_(envMapMutex_);
        env_[key] = value;
    }
}

void FileTelemetry::deleteAppEnvironmentValue(const std::string& key) {
    {
        std::scoped_lock<std::mutex> scopedLock_(envMapMutex_);
        env_.erase(key);
    }
}

void FileTelemetry::writeToFile(std::string_view data) {
    auto t = std::time(nullptr);
    struct tm buf;
    ::localtime_r(&t, &buf);
    std::scoped_lock<std::mutex> scopedLock(writeMutex_);
    outputFile_ << '\n'
                << std::put_time(&buf, "%d-%m-%Y %H:%M:%S") << ": " << data << std::flush;
}

std::shared_ptr<const YandexIO::LatencyData> FileTelemetry::createLatencyPoint() {
    return std::make_shared<const YandexIO::LatencyData>();
}

void FileTelemetry::reportLatency(std::shared_ptr<const YandexIO::LatencyData> /*latencyData*/, const std::string& /*eventName*/) {
}

void FileTelemetry::reportLatency(std::shared_ptr<const YandexIO::LatencyData> /*latencyData*/, const std::string& /*eventName*/, const std::string& /*eventJson*/) {
}

void FileTelemetry::setRateLimiter(const std::string& /*config*/) {
}

void FileTelemetry::requestUUID(UUIDCallback cb) {
    cb(uuid_);
}

void FileTelemetry::setSenderName(const std::string& /*name*/) {
}

ITelemetry::IParams* FileTelemetry::params() {
    static NullParams params;
    return &params;
}
