#include <yandex_io/libs/telemetry/named_latency.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/json_utils/json_utils.h>

using namespace YandexIO;

NamedLatency::NamedLatency(const std::shared_ptr<ITelemetry>& telemetry)
    : telemetry_(telemetry)
{
}

void NamedLatency::createLatencyPoint(const std::string& latencyPointName) {
    auto point = telemetry_->createLatencyPoint();
    std::scoped_lock lock(mutex_);
    auto [iter, inserted] = points_.emplace(latencyPointName, point);
    if (!inserted) {
        YIO_LOG_WARN("Latency point " << latencyPointName << " already exists! Replaced by new one");
        iter->second = std::move(point);
    }
}

void NamedLatency::alertAbsentPoint(const std::string& latencyPointName) {
    YIO_LOG_WARN("Trying to report absent latency point " << latencyPointName);
}

void NamedLatency::reportLatency(const std::string& latencyPointName, const std::string& eventName, bool removePoint) {
    reportLatencyBuilder(latencyPointName, eventName, removePoint, []() { return Json::Value(Json::objectValue); });
}

void NamedLatency::reportLatency(const std::string& latencyPointName, const std::string& eventName, bool removePoint, const std::string& jsonEvent) {
    reportLatencyBuilder(latencyPointName, eventName, removePoint, [&jsonEvent]() { return quasar::tryParseJsonOrEmpty(jsonEvent); });
}

void NamedLatency::reportLatency(const std::string& latencyPointName, const std::string& eventName, bool removePoint, Json::Value jsonEvent) {
    reportLatencyBuilder(latencyPointName, eventName, removePoint, [jsonEvent = std::move(jsonEvent)]() { return jsonEvent; });
}

void NamedLatency::reportEvent(const std::string& eventName, Json::Value jsonEvent) {
    telemetry_->reportEvent(eventName, quasar::jsonToString(jsonEvent));
}
