#pragma once

#include <string>
#include <unordered_map>

#include <yandex_io/libs/telemetry/latency_data.h>

namespace YandexIO {
    /** Abstract telemetry interface.
     *
     * Instances of this interface implement event-reporting channels for the client
     * code.
     */
    class ITelemetry {
    public:
        using Flags = uint64_t;

        using UUIDCallback = std::function<void(const std::string&)>;

        static const Flags SKIP_DATABASE = 1;

        virtual ~ITelemetry();

        virtual void reportEvent(const std::string& event, ITelemetry::Flags flags = 0) = 0;
        virtual void reportEvent(const std::string& event, const std::string& eventJson, ITelemetry::Flags flags = 0) = 0;

        virtual void reportError(const std::string& error, ITelemetry::Flags flags = 0) = 0;
        // will produce java-style error
        virtual void reportError(const std::string& error, const std::string& errorValue, ITelemetry::Flags flags = 0) = 0;

        virtual void reportLogError(const std::string& message, const std::string& sourceFileName, size_t sourceLine, const std::string& eventJson = {}) = 0;

        virtual void reportKeyValues(const std::string& eventName, const std::unordered_map<std::string, std::string>& keyValues, ITelemetry::Flags flags = 0) = 0;

        virtual std::shared_ptr<const LatencyData> createLatencyPoint() = 0;
        virtual void reportLatency(std::shared_ptr<const LatencyData> latencyData, const std::string& eventName) = 0;
        virtual void reportLatency(std::shared_ptr<const LatencyData> latencyData, const std::string& eventName, const std::string& eventJson) = 0;

        virtual void setRateLimiter(const std::string& config) = 0;

        virtual void putAppEnvironmentValue(const std::string& key, const std::string& value) = 0;
        virtual void deleteAppEnvironmentValue(const std::string& key) = 0;

        virtual void requestUUID(UUIDCallback cb) = 0;
        virtual void setSenderName(const std::string& name) = 0;

        class IParams {
        public:
            virtual ~IParams();

            enum class ConnectionType {
                UNKNOWN = 0,
                WIFI,
                ETHERNET
            };

            struct NetworkStatus {
                enum Status {
                    NOT_CONNECTED = 0,
                    CONNECTING,
                    CONNECTED_NO_INTERNET,
                    CONNECTED,
                    NOT_CHOSEN
                };

                Status status;
                ConnectionType type;
            };

            virtual void setNetworkStatus(NetworkStatus status) = 0;

            struct WifiNetwork {
                std::string mac;
                int32_t rssi;
                std::string ssid;
                bool isConnected;
            };

            virtual void setWifiNetworks(const std::vector<WifiNetwork>& networks) = 0;

            virtual void setLocation(double lat, double lon) = 0;

            virtual void setTimezone(const std::string& name, int32_t offsetSec) = 0;

            virtual void setConfig(const std::string& config) = 0;
        };

        virtual IParams* params() = 0;
    };

} /* namespace YandexIO */
