#pragma once

//
// Copyright (c) 2013 Juan Palacios juan.palacios.puyana@gmail.com
// Subject to the BSD 2-Clause License
// - see < http://opensource.org/licenses/BSD-2-Clause>
//

#pragma once

#include "steady_condition_variable.h"

#include <functional>
#include <iostream>
#include <mutex>
#include <queue>
#include <thread>

namespace quasar {

    template <typename T>
    class BlockingQueue {
    public:
        using Callback = std::function<void(size_t)>;

        BlockingQueue(int maxSize,
                      Callback onOverflow = Callback(),
                      Callback onBecomeNormal = Callback())
            : maxSize_(maxSize)
            , onOverflow_(std::move(onOverflow))
            , onBecomeNormal_(std::move(onBecomeNormal))
        {
        }

        void pop(T& item)
        {
            std::unique_lock<std::mutex> mlock(mutex_);
            cond_.wait(mlock, [this]() {
                return !queue_.empty();
            });
            item = std::move(queue_.front());
            queue_.pop();

            const int newSize = queue_.size();
            const auto maxSize = maxSize_;

            mlock.unlock();

            // become normal on maxSize - 1 to reduce oscillation processes
            if (newSize == maxSize - 1 && onBecomeNormal_) {
                onBecomeNormal_(newSize);
            }
        }

        bool tryPop(T& item)
        {
            std::unique_lock<std::mutex> mlock(mutex_);
            if (queue_.empty()) {
                return false;
            }
            item = std::move(queue_.front());
            queue_.pop();
            const int newSize = queue_.size();
            const auto maxSize = maxSize_;
            mlock.unlock();

            // become normal on maxSize - 1 to reduce oscillation processes
            if (newSize == maxSize - 1 && onBecomeNormal_) {
                onBecomeNormal_(newSize);
            }
            return true;
        }

        template <typename Duration>
        bool tryPop(T& item, const Duration duration)
        {
            std::unique_lock<std::mutex> mlock(mutex_);
            if (!cond_.wait_for(mlock, duration, [this]() {
                    return !queue_.empty();
                })) {
                return false;
            }
            item = std::move(queue_.front());
            queue_.pop();

            const int newSize = queue_.size();
            const auto maxSize = maxSize_;

            mlock.unlock();

            // become normal on maxSize - 1 to reduce oscillation processes
            if (newSize == maxSize - 1 && onBecomeNormal_) {
                onBecomeNormal_(newSize);
            }
            return true;
        }

        void push(T item)
        {
            std::unique_lock<std::mutex> mlock(mutex_);
            const int oldSize = queue_.size();
            queue_.push(std::move(item));
            const auto maxSize = maxSize_;
            cond_.notify_one();
            mlock.unlock();
            if (oldSize == maxSize && onOverflow_)
            {
                onOverflow_(oldSize + 1);
            }
        }

        bool tryPush(T item)
        {
            std::scoped_lock<std::mutex> mlock(mutex_);
            const int oldSize = queue_.size();
            bool itemAdded = false;
            if (oldSize < maxSize_)
            {
                queue_.push(std::move(item));
                itemAdded = true;
            }
            if (itemAdded) {
                cond_.notify_one();
            }
            return itemAdded;
        }

        size_t size() const {
            std::scoped_lock<std::mutex> lock(mutex_);
            return queue_.size();
        }

        void clear()
        {
            std::queue<T> q;
            std::scoped_lock<std::mutex> lock(mutex_);
            queue_.swap(q);
        }

        void setMaxSize(size_t size)
        {
            std::scoped_lock<std::mutex> lock(mutex_);
            maxSize_ = size;
        }

        BlockingQueue(const BlockingQueue&) = delete;            // disable copying
        BlockingQueue& operator=(const BlockingQueue&) = delete; // disable assignment

    private:
        std::queue<T> queue_;
        mutable std::mutex mutex_;
        quasar::SteadyConditionVariable cond_;

        int maxSize_;
        const Callback onOverflow_;
        const Callback onBecomeNormal_;
    };

} // namespace quasar
