#pragma once

#include "blocking_queue.h"
#include "i_callback_queue.h"
#include "steady_condition_variable.h"

#include <queue>

namespace quasar {
    class CallbackQueue: public ICallbackQueue {
    public:
        using Callback = std::function<void(size_t queueSize)>;
        /*
         * Use yandex_io/libs/base/NamedCallbackQueue instead
         * to be able to detect queue overflow properly
         */
        CallbackQueue(size_t maxSize = 1000, Callback onOverflow = nullptr, Callback onBecomeNormal = nullptr);
        ~CallbackQueue();
        std::string name() const override;
        void add(std::function<void()> callback) override;
        void add(std::function<void()> callback, Lifetime::Tracker tracker) override;
        bool tryAdd(std::function<void()> callback) override;
        bool tryAdd(std::function<void()> callback, Lifetime::Tracker tracker) override;

        void addDelayed(std::function<void()> callback, std::chrono::milliseconds timeOut) override;
        void addDelayed(std::function<void()> callback, std::chrono::milliseconds timeOut, Lifetime::Tracker tracker) override;
        void wait(AwatingType awatingType = AwatingType::ALTRUIST) override;
        bool isWorkingThread() const noexcept override;
        size_t size() const override;
        void setMaxSize(size_t size) override;
        void clear();

        /**
         * Must be called before captured objects are destructed!
         */
        void destroy();

    protected:
        virtual void onException(std::exception_ptr exptr) noexcept;

    private:
        void mainLoop();
        void timeOutedCallbacksLoop();

    private:
        BlockingQueue<std::function<void()>> queue_;
        std::thread queueThread_;

        using TimeOutedCallbacksPair = std::pair<std::chrono::steady_clock::time_point, std::function<void()>>;
        struct TimeOutedCallbacksCmp {
            bool operator()(const TimeOutedCallbacksPair& lhs, const TimeOutedCallbacksPair& rhs) const {
                return lhs.first > rhs.first;
            }
        };
        std::priority_queue<TimeOutedCallbacksPair, std::vector<TimeOutedCallbacksPair>, TimeOutedCallbacksCmp> timeOutedCallbacks_;
        std::thread timeOutedCallbacksThread_;
        SteadyConditionVariable wakeUpVar_;
        std::mutex mutex_;
        std::atomic<bool> stopped_{false};
    };
} // namespace quasar
