#pragma once
#include "lifetime.h"

#include <thread>
#include <sstream>

namespace quasar {

    class ICallbackQueue {
    public:
        virtual ~ICallbackQueue() = default;

        /**
         * @return unique callback queue name
         */
        virtual std::string name() const = 0;

        /**
         * @param callback non-null function. Lambdas should be used instead of std::bind
         *                 (check src/utils/test_bind_vs_lambda_vs_function)
         * @param lifetime it gives a guarantee that the callback will not be called if
         *                 lifetime is already expired, and also that while callback is executing,
         *                 lifetime cannot die and waits for the call to end
         * @param tracker  same as lifetime but for another situations
         * @brief execution is guaranteed
         */
        virtual void add(std::function<void()> callback) = 0;
        virtual void add(std::function<void()> callback, Lifetime::Tracker tracker) = 0;

        /**
         * @brief. Same as @add methods, but it does consider the size of CallbackQueue
         * @return true - added callback. false - failed to add callback
         */
        virtual bool tryAdd(std::function<void()> callback) = 0;
        virtual bool tryAdd(std::function<void()> callback, Lifetime::Tracker tracker) = 0;

        /**
         * @param callback non-null function. Lambdas should be used instead of std::bind
         *                 (check src/utils/test_bind_vs_lambda_vs_function)
         * @param timeOut - min pause before execution
         * @param lifetime it gives a guarantee that the callback will not be called if
         *                 lifetime is already expired, and also that while callback is executing,
         *                 lifetime cannot die and waits for the call to end
         * @param tracker  same as lifetime but for another situations
         * @brief execution is not guaranteed in case of object destruction
         */
        virtual void addDelayed(std::function<void()> callback, std::chrono::milliseconds timeOut) = 0;
        virtual void addDelayed(std::function<void()> callback, std::chrono::milliseconds timeOut, Lifetime::Tracker tracker) = 0;

        /**
         * @brief The call blocks the current thread and waits until all the tasks are completed
         */
        enum class AwatingType {
            EGOIST,   // Waits only for tasks that were scheduled at the time the wait method was called
            ALTRUIST, // Waits for all tasks in the queue to complete, even those that were
                      // scheduled after the method was called, but does not wait for pending tasks to complete
        };
        virtual void wait(AwatingType = AwatingType::ALTRUIST) = 0;

        /**
         * @brief Return true if calling from working thread
         */
        virtual bool isWorkingThread() const noexcept = 0;

        /**
         * @brief Returns current size of callback queue
         */
        virtual size_t size() const = 0;

        /**
         * @brief Sets maximum size of callback queue
         */
        virtual void setMaxSize(size_t size) = 0;
    };

    template <typename F>
    auto makeSafeCallback(F callback, Lifetime::Tracker tracker);
    template <typename F>
    auto makeSafeCallback(F callback, Lifetime::Tracker tracker, std::weak_ptr<ICallbackQueue> callbackQueue);

} // namespace quasar

#define Y_ENSURE_THREAD(callbackQueue)                                                                                                    \
    do {                                                                                                                                  \
        if ((callbackQueue) && (callbackQueue)->isWorkingThread()) {                                                                      \
        } else {                                                                                                                          \
            YIO_DEFINE_LOG_MODULE_IN_SCOPE("callback_queue");                                                                             \
            std::stringstream ss;                                                                                                         \
            ss << "Unexpected thread to call method " << __PRETTY_FUNCTION__ << " (callback queue is " << (callbackQueue)->name() << ")"; \
            YIO_LOG_ERROR_EVENT("CallbackQueue.EnsureThreadFail", ss.str());                                                              \
            throw std::logic_error(ss.str());                                                                                             \
        }                                                                                                                                 \
    } while (false)

#define Y_ENSURE_NOT_THREAD(callbackQueue)                                                                                                \
    do {                                                                                                                                  \
        if (!(callbackQueue) || !((callbackQueue)->isWorkingThread())) {                                                                  \
        } else {                                                                                                                          \
            YIO_DEFINE_LOG_MODULE_IN_SCOPE("callback_queue");                                                                             \
            std::stringstream ss;                                                                                                         \
            ss << "Unexpected thread to call method " << __PRETTY_FUNCTION__ << " (callback queue is " << (callbackQueue)->name() << ")"; \
            YIO_LOG_ERROR_EVENT("CallbackQueue.EnsureThreadFail", ss.str());                                                              \
            throw std::logic_error(ss.str());                                                                                             \
        }                                                                                                                                 \
    } while (false)

#include "i_callback_queue.inl"
