#pragma once

#include "i_callback_queue.h"

#include <chrono>

namespace quasar {

    class IPeriodicTask {
    public:
        virtual ~IPeriodicTask() = default;

        virtual bool expired() const noexcept = 0;
        virtual bool restart(std::chrono::milliseconds interval) noexcept = 0;
        virtual bool stop() noexcept = 0;
    };

    class IDelayedTask {
    public:
        virtual ~IDelayedTask() = default;

        virtual bool expired() const noexcept = 0;
        virtual bool stop() noexcept = 0;
    };

    class ITimerService {
    public:
        virtual ~ITimerService() = default;

        /**
         * @param interval - the timeout repetition period
         * @param callback - non-null function. Lambdas should be used instead of std::bind
         * @param lifetime - it gives a guarantee that the callback will not be called if
         *                  lifetime is already expired, and also that while callback is executing,
         *                  lifetime cannot die and waits for the call to end
         * @param callbackQueue - the queue in which this callback will be called
         * @brief The timer repeats the callback after the specified amount of time. If you do not save the link to the ITimer for cancellation,
         *        then the timer will repeat as long as the Lifetime and CallbackQueue are alive
         */
        virtual std::shared_ptr<IPeriodicTask> createPeriodicTask(std::chrono::milliseconds interval, std::function<void()> callback, const Lifetime::Tracker& tracker, std::weak_ptr<ICallbackQueue> callbackQueue) noexcept = 0;

        /**
         * @param interval - time after which callback will be called
         * @param callback - non-null function. Lambdas should be used instead of std::bind
         * @param lifetime - it gives a guarantee that the callback will not be called if
         *                  lifetime is already expired, and also that while callback is executing,
         *                  lifetime cannot die and waits for the call to end
         * @param callbackQueue - the queue in which this callback will be called
         * @brief The timer repeats the callback after the specified amount of time. If you do not save the link to the ITimer for cancellation,
         *        the timer will be executed as long as the Lifetime and CallbackQueue are alive
         */
        virtual std::shared_ptr<IDelayedTask> createDelayedTask(std::chrono::milliseconds interval, std::function<void()> callback, const Lifetime::Tracker& tracker, std::weak_ptr<ICallbackQueue> callbackQueue) noexcept = 0;
    };

    template <class Task>
    class TaskHolder {
    public:
        TaskHolder() = default;
        TaskHolder(const std::shared_ptr<Task>& /*task*/);
        TaskHolder(std::shared_ptr<Task>&& /*task*/);
        TaskHolder(TaskHolder&& /*other*/) noexcept;
        ~TaskHolder();
        TaskHolder& operator=(TaskHolder&& /*other*/) noexcept;
        TaskHolder& operator=(const std::shared_ptr<Task>& /*task*/);
        TaskHolder& operator=(std::shared_ptr<Task>&& /*task*/);

        const Task* operator->() const;
        Task* operator->();
        bool operator!() const;
        operator bool() const;

    private:
        std::shared_ptr<Task> task_;
    };

} // namespace quasar

#include "i_timer_service.inl"
