#include "lifetime.h"

namespace quasar {
    namespace {

        struct Soul {
            const Lifetime* lifetime{nullptr};
            std::atomic<bool> dying{false};

            Soul() = default;
            Soul(const Lifetime* l, bool d)
                : lifetime(l)
                , dying(d)
            {
            }
        };

        const auto DYING = std::make_shared<Soul>(nullptr, true);

    } // namespace

    const Lifetime Lifetime::immortal;

    Lifetime::Tracker::Tracker(std::nullptr_t)
    {
    }

    Lifetime::Tracker::Tracker(const Lifetime& lifetime)
        : Tracker(lifetime.tracker())
    {
    }

    Lifetime::Tracker::Tracker(std::weak_ptr<const void> soul, LifetimeTag /*tag*/)
        : soul_(std::move(soul))
        , lifetime_(true)
    {
    }

    bool Lifetime::Tracker::isValid() const noexcept {
        std::weak_ptr<const void> empty;
        return soul_.owner_before(empty) || empty.owner_before(soul_);
    }

    bool Lifetime::Tracker::expired() const noexcept {
        return soul_.expired();
    }

    std::shared_ptr<const void> Lifetime::Tracker::lock() const noexcept {
        auto refSoul = soul_.lock();
        if (lifetime_ && refSoul) {
            const Soul* soul = static_cast<const Soul*>(refSoul.get());
            if (soul->dying.load()) {
                return nullptr;
            }
        }
        return refSoul;
    }

    Lifetime::~Lifetime() {
        resetTracker();
    }

    Lifetime::Tracker Lifetime::tracker() const noexcept {
        std::lock_guard<std::mutex> lock(mutex_);
        if (!soul_)
        {
            soul_ = std::shared_ptr<Soul>(new Soul{this, false},
                                          [](const Soul* soul) {
                                              /* Пояснения:
                                               * Тут добавили lock для того чтоб notify_all не улетал в молоко. При некоторых
                                               * ситуациях у нас получается такой порядок вызовов:
                                               * -- мьютекс mutex_ заблокирован --
                                               * ...
                                               * (Поток 1): Если weakSoul.expired(), то выходи ничего не ждем
                                               * (Поток 2): Умерла последняя ссылка на soul_ (только сейчас weakSoul стал expired)
                                               * (Поток 2): notify_all (но никто не ждет)
                                               * (Поток 1): Раз weakSoul.expired() еще живой, то встаем на ожидание и разблокируем мьютес mutex_
                                               * ... Зависли, т.к. никто больше не пришлет notify_all()
                                               *
                                               * После добавления std::lock_guard lock(self->mutex_) получит такой порядок
                                               * Вариант 1.
                                               * -- мьютекс mutex_ заблокирован --
                                               * ...
                                               * (Поток 1): Если weakSoul.expired(), то выходи ничего не ждем
                                               * (Поток 2): Умерла последняя ссылка на soul_ (только сейчас weakSoul стал expired)
                                               * (Поток 2): Ждем захвата mutex_
                                               * (Поток 1): Раз weakSoul.expired() еще живой, то встаем на ожидание и разблокируем мьютес mutex_
                                               * (Поток 2): Дождались захвата mutex_
                                               * (Поток 2): notify_all (его ждет первый поток)
                                               * (Поток 1): Снялись с ожидания посшли проверять что там с weakSoul
                                               *
                                               * Вариант 2.
                                               * (Поток 2): Умерла последняя ссылка на soul_ (только сейчас weakSoul стал expired)
                                               * (Поток 2): Захватываем mutex_
                                               * (Поток 1): Ждем захвата mutex_
                                               * (Поток 2): notify_all (но никто не ждет)
                                               * (Поток 2): Отпустили mutex_
                                               * (Поток 1): Если weakSoul.expired(), то выходи ничего не ждем.
                                               * (Поток 1): Выходим без ожидания и разблокируем мьютес mutex_
                                               */
                                              std::lock_guard lock(soul->lifetime->mutex_);
                                              soul->lifetime->cv_.notify_all();
                                              delete soul;
                                          });
        }
        return Tracker(soul_, Tracker::LifetimeTag{});
    }

    void Lifetime::die() noexcept {
        if (this == &immortal)
        {
            return;
        }
        resetTracker();
    }

    long Lifetime::use_count() const noexcept {
        std::lock_guard<std::mutex> lock(mutex_);
        return soul_.use_count();
    }

    bool Lifetime::expired() const noexcept {
        std::lock_guard<std::mutex> lock(mutex_);
        return !soul_;
    }

    void Lifetime::resetTracker() noexcept {
        std::unique_lock<std::mutex> lock(mutex_);
        if (Soul* soul = static_cast<Soul*>(soul_.get())) {
            soul->dying = true; // mark original tracker as dying, this tracker can't be locked in future
        }
        std::weak_ptr<void> weakSoul = soul_;
        auto refSoul = soul_; // We guarantee that soul_ will not be destroyed inside a locked mutex
        soul_ = DYING;        // shoud replace with "fake" object to release ref of original tracker
        lock.unlock();

        refSoul.reset(); // release original tracker reference outside of mutex border

        lock.lock();
        cv_.wait(lock, [&] { return weakSoul.expired(); });
        soul_ = nullptr; // After totaly ding you can lock lifetime again
    }

} // namespace quasar
