#include "periodic_executor.h"

using namespace quasar;

PeriodicExecutor::PeriodicExecutor(std::function<void()> func, std::chrono::milliseconds period, PeriodicExecutor::PeriodicType type)
    : executeFunc_(std::move(func))
    , periodMs_(period.count())
{
    if (!executeFunc_) {
        throw std::runtime_error("Cannot run PeriodicExecutor with empty function");
    }

    switch (type) {
        case PeriodicType::ONE_SHOT: {
            executeThread_ = std::thread(&PeriodicExecutor::executeOneShot, this);
            break;
        }
        case PeriodicType::SLEEP_FIRST: {
            executeThread_ = std::thread(&PeriodicExecutor::sleepFirstLoop, this);
            break;
        }
        case PeriodicType::CALLBACK_FIRST: {
            executeThread_ = std::thread(&PeriodicExecutor::callbackFirstLoop, this);
            break;
        }
    }
}

std::chrono::milliseconds PeriodicExecutor::periodTime() const {
    return std::chrono::milliseconds(periodMs_.load());
}

void PeriodicExecutor::setPeriodTime(std::chrono::milliseconds period)
{
    periodMs_ = period.count();
}

void PeriodicExecutor::setNewCallback(std::function<void()> func)
{
    std::lock_guard<std::mutex> lock(mutex_);
    executeFunc_ = std::move(func);

    if (!executeFunc_) {
        throw std::runtime_error("Cannot run PeriodicExecutor with empty function");
    }
}

void PeriodicExecutor::executeNow()
{
    std::unique_lock<std::mutex> lock(mutex_);
    needExecute_ = true;
    lock.unlock();

    wakeupVar_.notify_one();
}

PeriodicExecutor::~PeriodicExecutor()
{
    std::unique_lock<std::mutex> lock(mutex_);
    stopped_ = true;
    wakeupVar_.notify_one();
    lock.unlock();
    executeThread_.join();
}

void PeriodicExecutor::executeOneShot()
{
    std::unique_lock<std::mutex> lock(mutex_);
    const auto now = std::chrono::steady_clock::now();
    const auto deadline = now + periodTime();
    wakeupVar_.wait_until(lock, deadline, [this]() { return needExecute_ || stopped_; });
    if (!stopped_) {
        lock.unlock();
        executeFunc_();
    }
}

void PeriodicExecutor::callbackFirstLoop()
{
    std::unique_lock<std::mutex> lock(mutex_);
    while (!stopped_)
    {
        lock.unlock();

        executeFunc_();

        lock.lock();

        const auto now = std::chrono::steady_clock::now();
        const auto deadline = now + periodTime();
        needExecute_ = false;
        wakeupVar_.wait_until(lock, deadline, [this]() { return needExecute_ || stopped_; });
    }
}

void PeriodicExecutor::sleepFirstLoop()
{
    std::unique_lock<std::mutex> lock(mutex_);
    while (!stopped_)
    {
        const auto now = std::chrono::steady_clock::now();
        const auto deadline = now + periodTime();
        needExecute_ = false;
        wakeupVar_.wait_until(lock, deadline, [this]() { return needExecute_ || stopped_; });
        if (stopped_) {
            return;
        }
        lock.unlock();

        executeFunc_();

        lock.lock();
    }
}
