#pragma once

#include "steady_condition_variable.h"

#include <atomic>
#include <chrono>
#include <functional>
#include <mutex>
#include <thread>

namespace quasar {

    class PeriodicExecutor {
    public:
        /*! Modes of PeriodicExecutor work */
        enum class PeriodicType {
            CALLBACK_FIRST, /*!< Callback will be called before the sleep */
            SLEEP_FIRST,    /*!< Callback will be called after sleeping for periodMs */
            ONE_SHOT        /*!< Callback will be called after sleeping for periodMs. After callback call PeriodicExecutor will not do anything */
        };

        /**
         * @brief Class that call callback function periodically in separate thread
         * @param func - Callback function to call
         * @param periodMs - Initial Period MS btw callback calls
         * @param type - Mode of PeriodicExecutor. @see PeriodicType
         */
        PeriodicExecutor(std::function<void()> func, std::chrono::milliseconds period, PeriodicType type = PeriodicType::CALLBACK_FIRST);
        /**
         * @brief Same as first constructor. Only input callback differ
         * @param peCallback - Struct that wrap the callback function which has a pointer to `this` as an argument.
         *               `this` will be passed as an argument. Input `this` pointer can be used safely change PeriodMs
         *               (by using setPeriodType method). `self` argument will be always valid, since last time callback can
         *               be called during the PeriodicExecutor destructor and then callback thread will be joined
         */

        struct PECallback {
            explicit PECallback(std::function<void(PeriodicExecutor*)> func)
                : callback(std::move(func))
            {
            }
            std::function<void(PeriodicExecutor*)> callback;
        };

        PeriodicExecutor(PECallback peCallback, std::chrono::milliseconds period, PeriodicType type)
            : PeriodicExecutor([peCallback, this]() {
                peCallback.callback(this);
            }, period, type)
        {
        }

        void executeNow();

        /**
         * @brief set Period ms time, which will be used in next iteration
         */
        void setPeriodTime(std::chrono::milliseconds period);
        std::chrono::milliseconds periodTime() const;

        void setNewCallback(std::function<void()> func);

        ~PeriodicExecutor();

    private:
        void executeOneShot();
        void sleepFirstLoop();
        void callbackFirstLoop();

        std::function<void()> executeFunc_;
        std::thread executeThread_;
        std::mutex mutex_;
        SteadyConditionVariable wakeupVar_;
        bool stopped_ = false;

        std::atomic<int64_t> periodMs_;

        bool needExecute_ = false;
    };

} // namespace quasar
