#pragma once

#include <yandex_io/libs/errno/errno_exception.h>

#include <util/system/yassert.h>

#include <condition_variable>
#include <functional>
#include <mutex>

#include <pthread.h>

namespace quasar {

    /**
     * std::condition_variable always uses std::chrono::system_clock (even if std::chrono::steady_clock is passed as argument int wait_until method.
     * SteadyConditionVariable guarantees to use monotonic clock and work correctly with NTP time changes
     */
    class SteadyConditionVariable {
    public:
        SteadyConditionVariable();

        template <typename Duration, typename UniqueLock>
        bool wait_until(UniqueLock& lock,
                        const std::chrono::time_point<std::chrono::steady_clock, Duration>& timePoint,
                        std::function<bool()> predicate)
        {
            Y_VERIFY(lock.owns_lock());
            Y_VERIFY(predicate);
            while (!predicate())
            {
                if (wait_until(lock, timePoint) == std::cv_status::timeout) {
                    return predicate();
                }
            }

            return true;
        }

        template <typename Duration, typename UniqueLock>
        std::cv_status wait_until(UniqueLock& lock,
                                  const std::chrono::time_point<std::chrono::steady_clock, Duration>& timePoint)
        {
            Y_VERIFY(lock.owns_lock());
            const auto duration = timePoint - std::chrono::steady_clock::now();
            const auto nanoseconds = std::chrono::duration_cast<std::chrono::nanoseconds>(duration);

            if (nanoseconds.count() < 0)
            {
                return std::cv_status::timeout;
            }

            struct timespec ts;
            clock_gettime(CLOCK_MONOTONIC, &ts);

            constexpr int32_t NANOSECONDS_IN_SECOND = 1000000000;

            ts.tv_nsec += nanoseconds.count() % NANOSECONDS_IN_SECOND;
            ts.tv_sec += nanoseconds.count() / NANOSECONDS_IN_SECOND + ts.tv_nsec / NANOSECONDS_IN_SECOND;
            ts.tv_nsec %= NANOSECONDS_IN_SECOND;

            const int res = pthread_cond_timedwait(&cond_, lock.mutex()->native_handle(), &ts);
            if (res != 0 && res != ETIMEDOUT)
            {
                throw ErrnoException(res, "Cannot pthread_cond_timedwait");
            }

            return (std::chrono::steady_clock::now() < timePoint
                        ? std::cv_status::no_timeout
                        : std::cv_status::timeout);
        }

        template <typename Duration, typename UniqueLock>
        std::cv_status wait_for(UniqueLock& lock, const Duration duration)
        {
            return wait_until(lock, std::chrono::steady_clock::now() + duration);
        }

        template <typename Duration, typename UniqueLock>
        bool wait_for(UniqueLock& lock,
                      const Duration duration, std::function<bool()> predicate)
        {
            return wait_until(lock, std::chrono::steady_clock::now() + duration, std::move(predicate));
        }

        template <typename UniqueLock>
        void wait(UniqueLock& lock)
        {
            Y_VERIFY(lock.owns_lock());
            const int res = pthread_cond_wait(&cond_, lock.mutex()->native_handle());
            if (res != 0) {
                throw ErrnoException(res, "Cannot pthread_cond_wait");
            }
        }

        template <typename UniqueLock>
        void wait(UniqueLock& lock, std::function<bool()> predicate)
        {
            Y_VERIFY(predicate);
            while (!predicate()) {
                wait(lock);
            }
        }

        void notify_one();
        void notify_all();

        ~SteadyConditionVariable();

    private:
        SteadyConditionVariable(const SteadyConditionVariable&) = delete;
        SteadyConditionVariable& operator=(const SteadyConditionVariable&) = delete;

        pthread_cond_t cond_;
        pthread_condattr_t condAttr_;
    };
} // namespace quasar
