#include "unique_callback.h"

#include <atomic>

namespace quasar {

    namespace {
        std::atomic<uint32_t> globalUniqueCallbackId{0};
    } // namespace

    struct UniqueCallback::UniqueCallbackData {
        std::weak_ptr<ICallbackQueue> callbackQueue;
        ReplaceType replaceType;

        std::mutex mutex;
        uint32_t uniqueCallbackId{0};
        std::chrono::steady_clock::time_point fireTime;
        Lifetime::Tracker tracker;
        std::function<void()> callback;
    };

    UniqueCallback::UniqueCallback(const std::shared_ptr<ICallbackQueue>& callbackQueue, ReplaceType replaceType)
        : data_(std::make_shared<UniqueCallbackData>())
    {
        data_->callbackQueue = callbackQueue;
        data_->replaceType = replaceType;
    }

    void UniqueCallback::execute(std::function<void()> callback, Lifetime::Tracker tracker) noexcept {
        executeDelayed(std::move(callback), std::chrono::milliseconds{0}, std::move(tracker));
    }

    void UniqueCallback::executeImmediately(const std::function<void()>& callback)
    {
        auto callbackQueue = data_->callbackQueue.lock();
        if (!callbackQueue) {
            throw std::logic_error("Undefined callbackQueue");
        }
        if (!callbackQueue->isWorkingThread()) {
            throw std::logic_error("UniqueCallback: Invalid call thread: " + callbackQueue->name());
        }

        reset();
        if (callback) {
            callback();
        }
    }

    void UniqueCallback::executeDelayed(std::function<void()> callback, std::chrono::milliseconds timeOut, Lifetime::Tracker tracker) noexcept {
        std::lock_guard lock(data_->mutex);
        bool replaceFirst = (data_->replaceType == ReplaceType::REPLACE_FIRST && timeOut.count() == 0);
        if (auto callbackQueue = data_->callbackQueue.lock()) {
            auto fireTime = std::chrono::steady_clock::now() + timeOut;
            if (!callback) {
                data_->uniqueCallbackId = 0;
                data_->fireTime = std::chrono::steady_clock::time_point{};
                data_->tracker = Lifetime::Tracker{};
                data_->callback = nullptr;
            } else if (replaceFirst && data_->uniqueCallbackId && data_->callback && data_->fireTime == std::chrono::steady_clock::time_point{}) {
                data_->tracker = tracker;
                data_->callback = std::move(callback);
            } else {
                if (data_->uniqueCallbackId && timeOut.count() && replaceFirst) {
                    if (data_->fireTime < fireTime) {
                        timeOut = std::chrono::duration_cast<std::chrono::milliseconds>(timeOut - (fireTime - data_->fireTime));
                        fireTime = data_->fireTime;
                    }
                }
                uint32_t uniqueCallbackId = ++globalUniqueCallbackId;
                data_->uniqueCallbackId = uniqueCallbackId;
                data_->fireTime = (timeOut.count() ? fireTime : std::chrono::steady_clock::time_point{});
                data_->tracker = tracker;
                data_->callback = std::move(callback);
                auto lambda =
                    [uniqueCallbackId, data = data_]()
                {
                    std::unique_lock llock(data->mutex);
                    if (data->uniqueCallbackId != uniqueCallbackId) {
                        return;
                    }
                    auto callback = std::move(data->callback);
                    auto tracker = std::move(data->tracker);
                    data->uniqueCallbackId = 0;
                    llock.unlock();
                    if (auto tlock = tracker.lock()) {
                        callback();
                    }
                };
                if (timeOut.count()) {
                    callbackQueue->addDelayed(std::move(lambda), timeOut, tracker);
                } else {
                    callbackQueue->add(std::move(lambda), tracker);
                }
            }
        }
    }

    void UniqueCallback::reset() noexcept {
        std::lock_guard lock(data_->mutex);
        data_->uniqueCallbackId = 0;
        data_->tracker = Lifetime::Tracker{};
        data_->callback = nullptr;
    }

    bool UniqueCallback::isScheduled() const noexcept {
        std::lock_guard lock(data_->mutex);
        return data_->uniqueCallbackId != 0;
    }

} // namespace quasar
