#pragma once
#include "i_callback_queue.h"

namespace quasar {

    class UniqueCallback {
    public:
        enum class ReplaceType {
            REPLACE_FIRST, // If this callback has already been added to the execution queue, then the new value will replace the existing one
            INSERT_BACK,   // If this callback has already been added to the queue for execution, then it will not be executed, and a new one will be added to the end of the queue

            // NOTE: executeDelayed always works as INSERT_BACK
        };
        UniqueCallback(const std::shared_ptr<ICallbackQueue>& callbackQueue, ReplaceType replaceType = ReplaceType::REPLACE_FIRST);

        /**
         * @brief: The task execution will be scheduled in the callbackQueue according to the ReplaceType
         */
        void execute(std::function<void()> callback, Lifetime::Tracker tracker) noexcept;

        /**
         * @brief: The method can only be called from the callbackQueue thread. The task will be executed synchronously,
         *         any other tasks will be canceled. If the method is called on the wrong thread, an exception will be
         *         thrown.
         */
        void executeImmediately(const std::function<void()>& callback);

        /**
         * @brief: The task will be added to the execution queue after the specified time interval. The rules for adding
         *         ReplaceType are preserved.
         */
        void executeDelayed(std::function<void()> callback, std::chrono::milliseconds timeOut, Lifetime::Tracker tracker) noexcept;

        /**
         * @brief: Cancel all scheduled and delayed tasks
         */
        void reset() noexcept;

        bool isScheduled() const noexcept;

    private:
        struct UniqueCallbackData;
        std::shared_ptr<UniqueCallbackData> data_;
    };

} // namespace quasar
