#include "time_trigger.h"
#include <util/datetime/systime.h>

using namespace quasar;

TimeCounter::TimeCounter()
    : count(0)
    , updated(std::chrono::steady_clock::now())
{
}

bool TimeCounter::updateImpl(unsigned newValue, TimePoint now, const TriggerConfig& cfg, bool lowOrHigh) {
    auto secsPassed = std::chrono::duration_cast<std::chrono::seconds>(now - updated);
    bool triggered = false;
    if (secsPassed >= cfg.interval) {
        if (cfg.interval > std::chrono::seconds(0)) {
            float normDiff = newValue >= count ? newValue - count : newValue; // handle reseted counter
            if (secsPassed > cfg.interval) {
                normDiff *= std::chrono::seconds(cfg.interval).count(); // extra cast to avoid problems with changes in TriggerConfig type
                normDiff /= secsPassed.count();
            }
            if (lowOrHigh) {
                triggered = normDiff > cfg.limit;
            } else {
                triggered = normDiff < cfg.limit;
            }
        }
        count = newValue;
        updated = now;
    }
    return triggered;
}

bool TooLowTrigger::update(unsigned newValue, TimePoint now, const TriggerConfig& cfg) {
    return updateImpl(newValue, now, cfg, false);
}

bool TooHighTrigger::update(unsigned newValue, TimePoint now, const TriggerConfig& cfg) {
    return updateImpl(newValue, now, cfg, true);
}

FalseTooLong::FalseTooLong()
    : updated(std::chrono::steady_clock::now())
{
}

void FalseTooLong::update(bool newValue, TimePoint now) {
    if (value || newValue) {
        updated = now;
    }
    value = newValue;
}

bool FalseTooLong::check(TimePoint now, std::chrono::seconds maxInterval) const {
    return !value && maxInterval != std::chrono::seconds(0) && (now - updated) > maxInterval;
}

/**********************************/

OnceADayTrigger::OnceADayTrigger()
    : lastTriggered(std::chrono::system_clock::now() - std::chrono::hours(24))
{
}

void OnceADayTrigger::changeTime(std::chrono::minutes newValue) {
    triggerTime = newValue;
}

bool OnceADayTrigger::check(std::chrono::system_clock::time_point now) {
    if (now - lastTriggered >= std::chrono::hours(24)) {
        const time_t epoch = std::chrono::system_clock::to_time_t(now);
        struct tm localTime;
        localtime_r(&epoch, &localTime);
        const auto minutesFromMidnight = std::chrono::minutes(std::chrono::hours(localTime.tm_hour)) + std::chrono::minutes(localTime.tm_min);
        if (minutesFromMidnight >= triggerTime) {
            lastTriggered = now;
            return true;
        }
    }
    return false;
}
