#pragma once

#include <websocketpp/client.hpp>
#include <websocketpp/endpoint.hpp>
#include <websocketpp/transport/asio/security/base.hpp>

#include <sstream>
#include <string>

namespace quasar {

    namespace Websocket {

        enum class StatusCode {
            BLANK = (int)websocketpp::close::status::blank,
            OMIT_HANDSHAKE = (int)websocketpp::close::status::omit_handshake,
            FORCE_TCP_DROP = (int)websocketpp::close::status::force_tcp_drop,
            NORMAL = (int)websocketpp::close::status::normal,
            GOING_AWAY = (int)websocketpp::close::status::going_away,
            PROTOCOL_ERROR = (int)websocketpp::close::status::protocol_error,
            UNSUPPORTED_DATA = (int)websocketpp::close::status::unsupported_data,
            NO_STATUS = (int)websocketpp::close::status::no_status,
            ABNORMAL_CLOSE = (int)websocketpp::close::status::abnormal_close,
            INVALID_PAYLOAD = (int)websocketpp::close::status::invalid_payload,
            POLICY_VIOLATION = (int)websocketpp::close::status::policy_violation,
            MESSAGE_TOO_BIG = (int)websocketpp::close::status::message_too_big,
            EXTENSION_REQUIRED = (int)websocketpp::close::status::extension_required,
            INTERNAL_ENDPOINT_ERROR = (int)websocketpp::close::status::internal_endpoint_error,
            SERVICE_RESTART = (int)websocketpp::close::status::service_restart,
            TRY_AGAIN_LATER = (int)websocketpp::close::status::try_again_later,
            BAD_GATEWAY = (int)websocketpp::close::status::bad_gateway,
            TLS_HANDSHAKE = (int)websocketpp::close::status::tls_handshake,
            SUBPROTOCOL_ERROR = (int)websocketpp::close::status::subprotocol_error,
            INVALID_SUBPROTOCOL_DATA = (int)websocketpp::close::status::invalid_subprotocol_data,

            INVALID_TOKEN = 4000,
            CLIENT_PONG_TIMEOUT = 4001,
            USER_DISCONNECT = 4002,
            SETTINGS_CHANGED = 4003,
        };

        enum class Error {
            BLANK = 0,
            SECURITY = (int)websocketpp::transport::asio::socket::error::security,
            SOCKET = (int)websocketpp::transport::asio::socket::error::socket,
            INVALID_STATE = (int)websocketpp::transport::asio::socket::error::invalid_state,
            INVALID_TLS_CONTEXT = (int)websocketpp::transport::asio::socket::error::invalid_tls_context,
            TLS_HANDSHAKE_TIMEOUT = (int)websocketpp::transport::asio::socket::error::tls_handshake_timeout,
            PASS_THROUGH = (int)websocketpp::transport::asio::socket::error::pass_through,
            MISSING_TLS_INIT_HANDLER = (int)websocketpp::transport::asio::socket::error::missing_tls_init_handler,
            TLS_HANDSHAKE_FAILED = (int)websocketpp::transport::asio::socket::error::tls_handshake_failed,
            TLS_FAILED_SNI_HOSTNAME = (int)websocketpp::transport::asio::socket::error::tls_failed_sni_hostname
        };

        struct ConnectionInfo {
            ConnectionInfo() = default;
            struct CloseInfo {
                StatusCode closeCode = StatusCode::BLANK;
                std::string closeReason;
            };
            CloseInfo local;
            CloseInfo remote;
            Error error = Error::BLANK;
            int httpResponseCode;

            std::string toString() const {
                std::stringstream message;
                message << "Remote close code: " << (int)remote.closeCode
                        << ", remote close reason: " << remote.closeReason
                        << "; local close code: " << (int)local.closeCode
                        << "; local close reason: " << local.closeReason
                        << "; error: " << (int)error
                        << "; http response code: " << httpResponseCode;
                return message.str();
            }
        };

        template <class ConnectionPtr>
        ConnectionInfo getConnectionInfo(ConnectionPtr connectionPtr, const websocketpp::connection_hdl& /* hdl */) {
            ConnectionInfo connectionInfo;
            connectionInfo.local.closeCode = static_cast<StatusCode>(connectionPtr->get_local_close_code());
            connectionInfo.local.closeReason = connectionPtr->get_local_close_reason();

            connectionInfo.remote.closeCode = static_cast<StatusCode>(connectionPtr->get_remote_close_code());
            connectionInfo.remote.closeReason = connectionPtr->get_remote_close_reason();

            connectionInfo.error = static_cast<Error>(connectionPtr->get_ec().value());
            connectionInfo.httpResponseCode = connectionPtr->get_response_code();

            return connectionInfo;
        }

    } // namespace Websocket

} // namespace quasar
