#include "websocket_client.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <cstdlib>
#include <ctime>
#include <memory>
#include <thread>

using namespace quasar;
using namespace quasar::Websocket;
using namespace std::chrono;
using namespace websocketpp::log;

namespace {
    std::map<std::string, level> channelNameToElevel = {
        {"devel", (level)elevel::devel},
        {"library", (level)elevel::library},
        {"info", (level)elevel::info},
        {"warning", (level)elevel::warn},
        {"error", (level)elevel::rerror},
        {"fatal", (level)elevel::fatal}};

    std::map<std::string, level> channelNameToAlevel = {
        {"connect", (level)alevel::connect},
        {"disconnect", (level)alevel::disconnect},
        {"control", (level)alevel::control},
        {"frame_header", (level)alevel::frame_header},
        {"frame_payload", (level)alevel::frame_payload},
        {"message_header", (level)alevel::message_header},
        {"message_payload", (level)alevel::message_payload},
        {"endpoint", (level)alevel::endpoint},
        {"debug_handshake", (level)alevel::debug_handshake},
        {"debug_close", (level)alevel::debug_close},
        {"devel", (level)alevel::devel},
        {"application", (level)alevel::app},
        {"http", (level)alevel::http},
        {"fail", (level)alevel::fail},
    };
    class CertificateError: public std::exception {
    };
} // namespace

void WebsocketClient::DisposableClient::Settings::TLS::ensureCorrect() const {
    // if not disabled only one of (crtFilePath, crtBuffer) should be set
    if (disabled) {
        return;
    }
    if (crtFilePath.empty() != crtBuffer.empty()) {
        return;
    }
    throw std::runtime_error("Incorrect TLSSettings. You should specify either one of "
                             "crtFilePath and crtBuffer or disable tls.");
}

std::shared_ptr<WebsocketClient::DisposableClient> WebsocketClient::DisposableClient::create(std::string connectUrl,
                                                                                             WebsocketClient::DisposableClient::Settings settings,
                                                                                             std::shared_ptr<YandexIO::ITelemetry> telemetry) {
    auto sptr = std::shared_ptr<DisposableClient>(new DisposableClient(std::move(telemetry)));
    sptr->init(std::move(connectUrl), std::move(settings));
    return sptr;
}

WebsocketClient::DisposableClient::DisposableClient(std::shared_ptr<YandexIO::ITelemetry> telemetry)
    : telemetry_(std::move(telemetry))
{
}

void WebsocketClient::DisposableClient::init(std::string connectUrl, Settings settings) {
    wthis_ = shared_from_this();

    endpointUrl_ = std::move(connectUrl);
    settings_ = std::move(settings);

    settings_.tls.ensureCorrect();
    connected_ = false;

    client_.init_asio();
    client_.start_perpetual();

    // dont log frame headers and bodies
    client_.clear_access_channels(websocketpp::log::alevel::frame_payload);
    client_.clear_access_channels(websocketpp::log::alevel::frame_header);

    client_.set_message_handler(std::bind(&DisposableClient::onMsg, this, std::placeholders::_1, std::placeholders::_2));
    client_.set_close_handler(std::bind(&DisposableClient::onClose, this, std::placeholders::_1));
    client_.set_open_handler(std::bind(&DisposableClient::onOpen, this, std::placeholders::_1));
    client_.set_fail_handler(std::bind(&DisposableClient::onFail, this, std::placeholders::_1));
    client_.set_tls_init_handler([=](websocketpp::connection_hdl /*hdl*/) {
        return DisposableClient::tlsInitHandler(endpointUrl_, settings_);
    });

    client_.set_ping_handler(std::bind(&DisposableClient::onPing, this, std::placeholders::_1, std::placeholders::_2));
    client_.set_pong_handler(std::bind(&DisposableClient::onPong, this, std::placeholders::_1, std::placeholders::_2));

    clientThread_ = std::thread(&DisposableClient::runWebsocketppThread, this);
    YIO_LOG_DEBUG("new DisposableClient " << (int*)this);
}

WebsocketClient::DisposableClient::~DisposableClient() {
    YIO_LOG_DEBUG("Destructing DisposableClient.. " << (int*)this);
    std::unique_lock lock(mutex_);
    client_.stop_perpetual();
    client_.stop();
    lock.unlock();
    {
        std::lock_guard<std::mutex> guard(periodicMutex_);
        lastPongCheckerPtr_.reset();
    }
    /*
     * FIXME: join is used because deadlock sometimes occurs.
     */
    joinClientThreadWithTimeout(seconds(30));
    YIO_LOG_DEBUG("Destructing DisposableClient.. Done for " << (int*)this);
}

void WebsocketClient::DisposableClient::joinClientThreadWithTimeout(std::chrono::milliseconds timeoutMs) {
    YIO_LOG_DEBUG("joining websocketpp...");
    auto latencyPoint = telemetry_->createLatencyPoint();
    auto status = websocketppClientThreadJoinedPromise_.get_future().wait_for(timeoutMs);
    telemetry_->reportLatency(latencyPoint, "websocketClientThreadJoining.done");
    if (status != std::future_status::ready) {
        YIO_LOG_DEBUG("clientThread_ is not joined after timeout");
        telemetry_->reportEvent("WebsocketClientDeadLocked");

        /*
         * We sleep before abort to give Telemetry a chance
         * to send the deadlock event.
         */
        std::this_thread::sleep_for(seconds(5));
        /*
         * abort() is used to generate minidump.
         */
        abort();
    }
    clientThread_.join();
    YIO_LOG_DEBUG("joining websocketpp... done");
}

void WebsocketClient::DisposableClient::runWebsocketppThread() {
    /*
     * client_.run() method blocks until websocket is explicitly
     * stopped via client_.stop_perpetual() and client_.stop().
     * run() is called only once per each DisposableClient.
     */
    client_.run();
    /*
     * This promise indicates client_.run() method is done
     * and websocket is stopped.
     */
    websocketppClientThreadJoinedPromise_.set_value();
};

bool WebsocketClient::DisposableClient::send(const std::string& msg) {
    std::unique_lock lock(mutex_);
    error_code ec;
    try {
        client_.send(connection_, msg, websocketpp::frame::opcode::text, ec);
    } catch (websocketpp::exception& exception) {
        YIO_LOG_DEBUG("websocket exception: " << exception.what());
    }
    return !ec;
}

bool WebsocketClient::DisposableClient::sendBinary(const std::string& msg) {
    std::unique_lock lock(mutex_);
    error_code ec;
    try {
        client_.send(connection_, msg, websocketpp::frame::opcode::binary, ec);
    } catch (websocketpp::exception& exception) {
        YIO_LOG_DEBUG("websocket exception: " << exception.what());
    }
    return !ec;
}

void WebsocketClient::DisposableClient::connectAsyncWithTimeout(size_t timeoutMs) {
    std::unique_lock lock(mutex_);

    client_.set_open_handshake_timeout(timeoutMs);
    try {
        websocketpp::lib::error_code error_code;
        connection_.reset();
        connection_ = client_.get_connection(endpointUrl_, error_code);
        if (error_code) {
            YIO_LOG_DEBUG("could not create connection to " << endpointUrl_ << ": " + error_code.message());
        } else {
            for (const auto& entry : settings_.connectionHeaders) {
                connection_->append_header(entry.first, entry.second);
            }
            client_.connect(connection_);
            YIO_LOG_INFO("Connection " << connection_.get() << " started via websocket");
        }
    } catch (CertificateError&) {
        ConnectionInfo connectionInfo;
        connectionInfo.error = Websocket::Error::TLS_HANDSHAKE_FAILED;
        if (onFailure) {
            onFailure(wthis_, connectionInfo);
        }
    } catch (std::exception& e) {
        YIO_LOG_DEBUG("Exception while connecting: " << e.what());
    }
}

void WebsocketClient::DisposableClient::checkClientLastPong() {
    std::unique_lock lock(mutex_);
    try {
        if (connected_ && steady_clock::now() - lastPongTs_ > pingIntervalCoef_ * settings_.ping.interval) {
            client_.close(connection_, (websocketpp::close::status::value)Websocket::StatusCode::CLIENT_PONG_TIMEOUT,
                          "no pong for long");
        } else {
            client_.ping(connection_, "client_ping");
        }
    } catch (const websocketpp::exception& exception) {
        YIO_LOG_DEBUG("websocketpp::exception: " << exception.what());
    }
}

void WebsocketClient::DisposableClient::onClose(websocketpp::connection_hdl hdl) {
    auto connectionInfo = getConnectionInfo(client_.get_con_from_hdl(hdl), hdl);
    std::unique_lock lock(mutex_);
    connected_ = false;
    lock.unlock();
    YIO_LOG_INFO("Websocket connection closed " << connectionInfo.toString());
    {
        std::lock_guard<std::mutex> guard(periodicMutex_);
        lastPongCheckerPtr_.reset();
    }
    if (onDisconnect) {
        onDisconnect(wthis_, connectionInfo);
    }
}

bool WebsocketClient::DisposableClient::onPing(websocketpp::connection_hdl hdl, const std::string& payload) {
    const auto connectionInfo = getConnectionInfo(client_.get_con_from_hdl(hdl), hdl);
    YIO_LOG_TRACE("Server Ping: " << connectionInfo.toString());
    if (onPingHandler) {
        return onPingHandler(wthis_, connectionInfo, payload);
    }
    return settings_.pong.enabled;
}

void WebsocketClient::DisposableClient::onPong(websocketpp::connection_hdl hdl, const std::string& payload) {
    {
        std::lock_guard lock(mutex_);
        lastPongTs_ = steady_clock::now();
    }

    if (onPongHandler) {
        const auto connectionInfo = getConnectionInfo(client_.get_con_from_hdl(hdl), hdl);
        YIO_LOG_TRACE("Server Pong: " << connectionInfo.toString());
        onPongHandler(wthis_, connectionInfo, payload);
    }
}

void WebsocketClient::DisposableClient::onOpen(websocketpp::connection_hdl /* hdl */) {
    YIO_LOG_INFO("Websocket connection opened");
    std::unique_lock lock(mutex_);
    connected_ = true;
    bool resetLastPongPeriodic = false;
    if (settings_.ping.enabled) {
        lastPongTs_ = steady_clock::now();
        resetLastPongPeriodic = true;
    }
    lock.unlock();
    if (resetLastPongPeriodic) {
        std::lock_guard<std::mutex> guard(periodicMutex_);
        lastPongCheckerPtr_ = std::make_unique<PeriodicExecutor>(
            std::bind(&WebsocketClient::DisposableClient::checkClientLastPong, this),
            settings_.ping.interval);
    }
    if (onConnect) {
        onConnect(wthis_);
    }
}

void WebsocketClient::DisposableClient::onFail(websocketpp::connection_hdl hdl) {
    auto connectionInfo = getConnectionInfo(client_.get_con_from_hdl(hdl), hdl);
    std::unique_lock lock(mutex_);
    connected_ = false;
    lock.unlock();

    YIO_LOG_INFO("Websocket connection fail " << connectionInfo.toString());
    {
        std::lock_guard<std::mutex> guard(periodicMutex_);
        lastPongCheckerPtr_.reset();
    }
    if (onFailure) {
        onFailure(wthis_, connectionInfo);
    }
}

void WebsocketClient::DisposableClient::onMsg(websocketpp::connection_hdl hdl, message_ptr msg) {
    YIO_LOG_TRACE("handler: " << hdl.lock().get());
    auto opcode = msg->get_opcode();
    if (opcode == websocketpp::frame::opcode::text && onMessage) {
        onMessage(wthis_, msg->get_payload());
    } else if (opcode == websocketpp::frame::opcode::binary && onBinaryMessage) {
        onBinaryMessage(wthis_, msg->get_payload());
    }
}

WebsocketClient::DisposableClient::context_ptr
WebsocketClient::DisposableClient::tlsInitHandler(const std::string& endpointUrl, const Settings& tlsSettings) {
    namespace asio = websocketpp::lib::asio;

    std::string hostname = getHostName(endpointUrl);
    context_ptr ctx = websocketpp::lib::make_shared<asio::ssl::context>(asio::ssl::context::tlsv12_client);
    ctx->set_options(asio::ssl::context::default_workarounds |
                     asio::ssl::context::single_dh_use);

    ctx->set_default_verify_paths();

    if (!tlsSettings.tls.disabled) {
        ctx->set_verify_mode(asio::ssl::verify_peer);
        if (tlsSettings.tls.verifyHostname) {
            ctx->set_verify_callback(asio::ssl::rfc2818_verification(hostname));
        }
        if (!tlsSettings.tls.crtFilePath.empty()) {
            ctx->load_verify_file(tlsSettings.tls.crtFilePath);
        } else if (!tlsSettings.tls.crtBuffer.empty()) {
            asio::error_code ec;
            ctx->add_certificate_authority(
                asio::const_buffer(tlsSettings.tls.crtBuffer.data(), tlsSettings.tls.crtBuffer.size()), ec);
            if (ec) {
                throw CertificateError();
            }
        } else {
            throw std::runtime_error("Incorrect tls settings");
        }
    } else {
        ctx->set_verify_mode(asio::ssl::verify_none);
    }
    return ctx;
}

std::string WebsocketClient::DisposableClient::getHostName(const std::string& connectUrl) {
    std::string prefixToErase = "wss://";
    std::string hostname = connectUrl.substr(prefixToErase.size(), connectUrl.size() - prefixToErase.size());
    std::size_t slash_pos = hostname.find('/');
    if (slash_pos != std::string::npos) {
        hostname = hostname.substr(0, hostname.find('/'));
    }
    return hostname;
}

/*
 * WebsocketClient
 */

WebsocketClient::WebsocketClient(std::shared_ptr<YandexIO::ITelemetry> telemetry)
    : telemetry_(std::move(telemetry))
{
    reconnectDelayMs_ = settings_.reconnect.delay.initMs;
}

WebsocketClient::~WebsocketClient()
{
    disconnectAsync();
    std::unique_lock lock(mutex_);
    stopped_ = true;
    interruptWait_ = true;
    lock.unlock();
    stateWakeUpVar_.notify_all();
}

WebsocketClient::Settings WebsocketClient::getSettings() const {
    return settings_;
}

void WebsocketClient::unsafeSend(const std::string& msg) {
    callbackQueue_.add([this, msg]() {
        if (state_ == State::CONNECTED) {
            clientPtr_->send(msg);
        } else {
            YIO_LOG_DEBUG("WebsocketClient is not connected. Message is dropped");
        }
    });
}

void WebsocketClient::unsafeSendBinary(const std::string& msg) {
    callbackQueue_.add([this, msg]() {
        if (state_ == State::CONNECTED) {
            clientPtr_->sendBinary(msg);
        } else {
            YIO_LOG_DEBUG("WebsocketClient is not connected. Message with size "
                          << msg.size() << " is dropped.");
        }
    });
}

void WebsocketClient::connectAsync(Settings settings) {
    if (settings.url.empty()) {
        throw std::runtime_error("No url for websocketClient");
    }
    // mutable lets move settings
    callbackQueue_.add([this, settings]() mutable {
        settings.tls.ensureCorrect();

        if (settings == settings_ && state_ == State::CONNECTED) {
            if (onConnect_) {
                onConnect_();
            }
        } else {
            if (state_ == State::CONNECTED) {
                ConnectionInfo connectionInfo;
                connectionInfo.local.closeCode = Websocket::StatusCode::SETTINGS_CHANGED;
                connectionInfo.local.closeReason = "Settings changed, reconnecting";
                if (onDisconnect_) {
                    onDisconnect_(connectionInfo);
                }
            }
            settings_ = std::move(settings);
            connectAsyncImpl();
        }
    });
}

/**
 * Must be called inside callbackQueueThread
 */
void WebsocketClient::connectAsyncImpl() {
    YIO_LOG_INFO("connectAsyncImpl url=" << settings_.url);
    state_ = State::CONNECTING;
    stateWakeUpVar_.notify_all();
    DisposableClient::Settings disposableWsSettings;
    disposableWsSettings.tls = settings_.tls;
    disposableWsSettings.ping = settings_.ping;
    disposableWsSettings.connectionHeaders = settings_.connectionHeaders;

    clientPtr_ = DisposableClient::create(settings_.url, disposableWsSettings, telemetry_);
    clientPtr_->setLogChannels(accessLogChannels_, errorLogChannels_);
    clientPtr_->onFailure = std::bind(&WebsocketClient::onFail, this, std::placeholders::_1, std::placeholders::_2);
    clientPtr_->onConnect = std::bind(&WebsocketClient::onOpen, this, std::placeholders::_1);
    clientPtr_->onDisconnect = std::bind(&WebsocketClient::onClose, this, std::placeholders::_1, std::placeholders::_2);
    clientPtr_->onMessage = std::bind(&WebsocketClient::onMessage, this, std::placeholders::_1, std::placeholders::_2);
    clientPtr_->onBinaryMessage = std::bind(&WebsocketClient::onBinaryMessage, this, std::placeholders::_1, std::placeholders::_2);
    clientPtr_->onPongHandler = std::bind(&WebsocketClient::onPong, this, std::placeholders::_1, std::placeholders::_2, std::placeholders::_3);
    clientPtr_->connectAsyncWithTimeout(settings_.connectTimeoutMs);
}

void WebsocketClient::connectSync(Settings settings) {
    settings.tls.ensureCorrect();
    connectAsync(settings);
    waitForState(State::CONNECTED);
}

bool WebsocketClient::connectSyncWithTimeout(Settings settings) {
    // drop interrupt flag before new connection attempt
    std::unique_lock lock(mutex_);
    interruptWait_ = false;
    lock.unlock();

    settings.tls.ensureCorrect();
    connectAsync(settings);
    return waitForStateWithTimeout(State::CONNECTED, settings.connectTimeoutMs);
}

void WebsocketClient::interruptWait() {
    std::unique_lock lock(mutex_);
    interruptWait_ = true;
    lock.unlock();
    stateWakeUpVar_.notify_all();
}

// TODO[katayad] remove onDone
void WebsocketClient::disconnectAsync(std::function<void()> onDone) {
    callbackQueue_.add([this, onDone]() {
        YIO_LOG_INFO("disconnectAsync");
        std::unique_lock lock(mutex_);
        interruptWait_ = true;
        lock.unlock();

        state_ = State::STOPPED;
        stateWakeUpVar_.notify_all();
        clientPtr_.reset();
        if (onDone) {
            onDone();
        }
    });
}

bool WebsocketClient::disconnectSyncWithTimeout(int timeoutMs) {
    // drop interrupt flag before new disconnect check
    std::unique_lock lock(mutex_);
    interruptWait_ = false;
    lock.unlock();

    disconnectAsync();
    return waitForStateWithTimeout(State::STOPPED, timeoutMs);
}

void WebsocketClient::onFail(std::weak_ptr<DisposableClient> wClientPtr, const ConnectionInfo& connectionInfo) {
    callbackQueue_.add([this, connectionInfo, wClientPtr]() {
        auto sClientPtr = wClientPtr.lock();
        if (sClientPtr) {
            std::unique_lock lock(mutex_);
            interruptWait_ = true;
            lock.unlock();

            state_ = State::CONNECTING;
            stateWakeUpVar_.notify_all();
            if (onFail_) {
                onFail_(connectionInfo);
            }
            if (settings_.reconnect.enabled) {
                delayedReconnect();
            }
        }
    });
}

void WebsocketClient::onPong(std::weak_ptr<DisposableClient> wClientPtr, const ConnectionInfo& /*info*/, const std::string& /*payload*/) {
    callbackQueue_.add([this, wClientPtr]() {
        if (auto sClientPtr = wClientPtr.lock()) {
            if (onPong_) {
                onPong_();
            }
        }
    });
}

void WebsocketClient::onClose(std::weak_ptr<DisposableClient> wClientPtr, const ConnectionInfo& connectionInfo) {
    callbackQueue_.add([this, connectionInfo, wClientPtr]() {
        auto sClientPtr = wClientPtr.lock();
        if (sClientPtr) {
            std::unique_lock lock(mutex_);
            interruptWait_ = true;
            lock.unlock();

            state_ = State::CONNECTING;
            stateWakeUpVar_.notify_all();
            if (onDisconnect_) {
                onDisconnect_(connectionInfo);
            }
            if (settings_.reconnect.enabled) {
                delayedReconnect();
            }
        }
    });
}

void WebsocketClient::onOpen(std::weak_ptr<DisposableClient> wClientPtr)
{
    callbackQueue_.add([this, wClientPtr]() {
        auto sClientPtr = wClientPtr.lock();
        if (sClientPtr) {
            state_ = State::CONNECTED;
            stateWakeUpVar_.notify_all();
            reconnectDelayMs_ = settings_.reconnect.delay.initMs;
            if (onConnect_) {
                onConnect_();
            }
        }
    });
}

void WebsocketClient::onMessage(std::weak_ptr<DisposableClient> wClientPtr, const std::string& message) {
    callbackQueue_.add([this, message, wClientPtr]() {
        auto sClientPtr = wClientPtr.lock();
        if (sClientPtr && onMessage_) {
            onMessage_(message);
        }
    });
}

void WebsocketClient::onBinaryMessage(std::weak_ptr<DisposableClient> wClientPtr, const std::string& message) {
    callbackQueue_.add([this, message, wClientPtr]() {
        auto sClientPtr = wClientPtr.lock();
        if (sClientPtr && onBinaryMessage_) {
            onBinaryMessage_(message);
        }
    });
}

/**
 * Must be called in callbackQueue thread
 */
void WebsocketClient::delayedReconnect() {
    std::weak_ptr<DisposableClient> wClientPtr = clientPtr_;
    callbackQueue_.addDelayed([this, wClientPtr]() {
        auto sClientPtr = wClientPtr.lock();
        if (sClientPtr) {
            if (state_ != State::STOPPED) {
                clientPtr_.reset();
                connectAsyncImpl();
            }
        }
    }, milliseconds(reconnectDelayMs_));
    YIO_LOG_DEBUG("Launched restart with delay: " << reconnectDelayMs_);

    const auto& delay = settings_.reconnect.delay;
    reconnectDelayMs_ = std::min(delay.offsetMs + reconnectDelayMs_ * delay.factor, delay.maxMs);
}

void WebsocketClient::setOnConnectHandler(OnConnect onConnect) {
    onConnect_ = std::move(onConnect);
}

void WebsocketClient::setOnMessageHandler(OnMessage onMessage) {
    onMessage_ = std::move(onMessage);
}

void WebsocketClient::setOnBinaryMessageHandler(OnMessage onMessage) {
    onBinaryMessage_ = std::move(onMessage);
}

void WebsocketClient::setOnDisconnectHandler(OnDisconnect onDisconnect) {
    onDisconnect_ = std::move(onDisconnect);
}

void WebsocketClient::setOnFailHandler(OnFail onFail) {
    onFail_ = std::move(onFail);
}

void WebsocketClient::setOnPongHandler(OnPong onPong) {
    onPong_ = std::move(onPong);
}

void WebsocketClient::waitForState(WebsocketClient::State state) {
    std::unique_lock lock(mutex_);
    stateWakeUpVar_.wait(lock, [this, state]() {
        return stopped_ || state_ == state;
    });
}

bool WebsocketClient::waitForStateWithTimeout(WebsocketClient::State state, int timeoutMs) {
    std::unique_lock lock(mutex_);
    stateWakeUpVar_.wait_for(lock, std::chrono::milliseconds(timeoutMs), [this, state]() {
        return stopped_ || interruptWait_ || state_ == state;
    });
    interruptWait_ = false;

    return state_ == state;
}

bool WebsocketClient::setLogChannels(const Json::Value& channelNames) {
    YIO_LOG_DEBUG("WebsocketClient::setLogChannels channelNames=" << jsonToString(channelNames));
    bool changed = false;

    level newChannels = getBitMaskFromChannelNames(channelNames, channelNameToAlevel);
    if (newChannels != accessLogChannels_) {
        accessLogChannels_ = newChannels;
        changed = true;
    }

    newChannels = getBitMaskFromChannelNames(channelNames, channelNameToElevel);
    if (newChannels != errorLogChannels_) {
        errorLogChannels_ = newChannels;
        changed = true;
    }

    YIO_LOG_DEBUG("WebsocketClient::accessLogChannels_=" << accessLogChannels_);
    YIO_LOG_DEBUG("WebsocketClient::errorLogChannels_=" << errorLogChannels_);

    if (clientPtr_ != nullptr) {
        clientPtr_->setLogChannels(accessLogChannels_, errorLogChannels_);
    }
    return changed;
}

level WebsocketClient::getBitMaskFromChannelNames(const Json::Value& channelNames,
                                                  const std::map<std::string, level>& channelToInt) {
    level bitmask = 0;
    for (size_t i = 0; i != channelNames.size(); ++i) {
        std::string channelName = getArrayElement(channelNames, i).asString();
        if (channelToInt.count(channelName)) {
            bitmask |= channelToInt.at(channelName);
        }
    }
    return bitmask;
}

void WebsocketClient::DisposableClient::setLogChannels(level accessLogChannels, level errorLogChannels) {
    client_.set_error_channels(errorLogChannels);
    client_.set_access_channels(accessLogChannels);
}

bool operator==(const WebsocketClient::DisposableClient::Settings::TLS& settings_a,
                const WebsocketClient::DisposableClient::Settings::TLS& settings_b) {
    return (settings_a.disabled == settings_b.disabled &&
            settings_a.verifyHostname == settings_b.verifyHostname &&
            settings_a.crtBuffer == settings_b.crtBuffer &&
            settings_a.crtFilePath == settings_b.crtFilePath);
}

bool operator==(const WebsocketClient::DisposableClient::Settings& settings_a,
                const WebsocketClient::DisposableClient::Settings& settings_b) {
    return (settings_a.tls == settings_b.tls &&
            settings_a.pong.enabled == settings_b.pong.enabled);
}

bool operator==(const WebsocketClient::Settings& settings_a, const WebsocketClient::Settings& settings_b) {
    return settings_a.url == settings_b.url && settings_a.tls == settings_b.tls;
}
