#pragma once

#include "loggable_asio_tls_client.h"
#include "websocket.h"

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/threading/periodic_executor.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <json/json.h>

#include <websocketpp/client.hpp>
#include <websocketpp/common/asio_ssl.hpp>
#include <websocketpp/logger/levels.hpp>

#include <cstdlib>
#include <ctime>
#include <future>
#include <iostream>
#include <map>

namespace quasar {

    /*
     * WebsocketClient is a class that manages websocket reconnections. It uses class DisposableClient
     * which actually keeps connection, checks certificates and etc. If DisposableClient loses connection
     * it can't be used for the new one, new DisposableClient must be created for new connection.
     *
     */

    class WebsocketClient {
    public:
        class DisposableClient: public std::enable_shared_from_this<DisposableClient> {
        public:
            using OnMessage = std::function<void(std::weak_ptr<DisposableClient> wthis, const std::string&)>;
            using OnConnect = std::function<void(std::weak_ptr<DisposableClient> wthis)>;
            using OnDisconnect = std::function<void(std::weak_ptr<DisposableClient> wthis, const Websocket::ConnectionInfo&)>;
            using OnFail = std::function<void(std::weak_ptr<DisposableClient> wthis, const Websocket::ConnectionInfo&)>;
            using OnPing = std::function<bool(std::weak_ptr<DisposableClient> wthis, const Websocket::ConnectionInfo&, const std::string& payload)>;
            using OnPong = std::function<void(std::weak_ptr<DisposableClient> wthis, const Websocket::ConnectionInfo&, const std::string& payload)>;

            using message_ptr = websocketpp::config::asio_tls_client::message_type::ptr;
            using websocket_client = websocketpp::client<LoggableAsioTlsClient>;
            using context_ptr = websocketpp::lib::shared_ptr<websocketpp::lib::asio::ssl::context>;
            using error_code = websocketpp::lib::error_code;

            struct Settings {
                struct TLS {
                    bool disabled = false;
                    bool verifyHostname = true;
                    std::string crtFilePath;
                    std::string crtBuffer;
                    void ensureCorrect() const;
                };
                struct Pong {
                    bool enabled = true;
                };
                struct Ping {
                    bool enabled = false;
                    std::chrono::milliseconds interval = std::chrono::seconds(5);
                } ping;
                TLS tls;
                Pong pong;

                std::map<std::string, std::string> connectionHeaders;
            };

        public:
            static std::shared_ptr<DisposableClient> create(std::string connectUrl, Settings settings, std::shared_ptr<YandexIO::ITelemetry> telemetry);
            ~DisposableClient();
            void connectAsyncWithTimeout(size_t timeoutMs = 5000);
            bool send(const std::string& msg);
            bool sendBinary(const std::string& msg);
            void setLogChannels(websocketpp::log::level accessLogChannels, websocketpp::log::level errorLogChannels);

            OnMessage onMessage;
            OnMessage onBinaryMessage;
            OnConnect onConnect;
            OnDisconnect onDisconnect;
            OnFail onFailure;
            OnPing onPingHandler;
            OnPong onPongHandler;

        private:
            explicit DisposableClient(std::shared_ptr<YandexIO::ITelemetry> telemetry);
            void init(std::string connectUrl, Settings settings);

            const std::shared_ptr<YandexIO::ITelemetry> telemetry_;
            std::weak_ptr<DisposableClient> wthis_;

            websocket_client client_;
            std::thread clientThread_;
            websocket_client::connection_ptr connection_;
            mutable std::mutex mutex_;

            std::string endpointUrl_;
            bool connected_;
            Settings settings_;

            std::promise<void> websocketppClientThreadJoinedPromise_;
            std::chrono::steady_clock::time_point lastPongTs_;
            const double pingIntervalCoef_ = 2.1;
            mutable std::mutex periodicMutex_; /* creating and destroying of lastPongCheckerPtr_ should be done under this mutex */
            /**
             * @note: lastPongCheckerPtr use DisposableClient::mutex_ inside, so avoid destructing it under this mutex
             */
            std::unique_ptr<PeriodicExecutor> lastPongCheckerPtr_;

            void onFail(websocketpp::connection_hdl hdl);
            void onClose(websocketpp::connection_hdl hdl);
            void onOpen(websocketpp::connection_hdl hdl);
            void onMsg(websocketpp::connection_hdl hdl, message_ptr msg);
            /**
             * @brief Handler for server Pings. Should return true if want to send Pong back.
             */
            bool onPing(websocketpp::connection_hdl hdl, const std::string& payload);
            void onPong(websocketpp::connection_hdl hdl, const std::string& payload);

            void joinClientThreadWithTimeout(std::chrono::milliseconds timeoutMs);

            static context_ptr tlsInitHandler(const std::string& endpointUrl, const Settings& settings);

            static std::string getHostName(const std::string& connectUrl);

            void runWebsocketppThread();

            void checkClientLastPong();
        };

    public:
        struct Settings {
            enum class Protocol { ipv4,
                                  ipv6 };

            DisposableClient::Settings::TLS tls;
            struct Reconnect {
                struct Delay {
                    /*
                     * delay is set to InitMs after each successful connection
                     */
                    int64_t initMs = 0;
                    /*
                     * Used in formula for every next unsuccessful try :
                     * delay = min(offsetMs + offsetCoef * delay, maxMs);
                     */
                    int64_t offsetMs = 1000;
                    int factor = 2;
                    int64_t maxMs = 30 * 1000;
                };
                bool enabled = true;
                Delay delay;
            };
            Reconnect reconnect;
            DisposableClient::Settings::Ping ping;
            std::string url;
            std::map<std::string, std::string> connectionHeaders;
            Protocol protocol;
            int connectTimeoutMs = 5000;
        };
        enum class State {
            STOPPED,
            CONNECTING,
            CONNECTED
        };

        using OnMessage = std::function<void(const std::string&)>;
        using OnConnect = std::function<void()>;
        using OnDisconnect = std::function<void(const Websocket::ConnectionInfo&)>;
        using OnFail = std::function<void(const Websocket::ConnectionInfo&)>;
        using OnPong = std::function<void()>;

        explicit WebsocketClient(std::shared_ptr<YandexIO::ITelemetry> telemetry);
        ~WebsocketClient();

        Settings getSettings() const;
        void connectAsync(Settings settings);
        void connectSync(Settings settings);
        bool connectSyncWithTimeout(Settings settings);
        void disconnectAsync(std::function<void()> onDone = nullptr);
        bool disconnectSyncWithTimeout(int timeoutMs);

        void unsafeSend(const std::string& msg);
        void unsafeSendBinary(const std::string& msg);

        void setOnConnectHandler(OnConnect onConnect);
        void setOnMessageHandler(OnMessage onMessage);
        void setOnBinaryMessageHandler(OnMessage onMessage);
        void setOnDisconnectHandler(OnDisconnect onDisconnect);
        void setOnFailHandler(OnFail onFail);
        void setOnPongHandler(OnPong onPong);

        void waitForState(State state);

        /*
         * Interruptable version of waitForState
         * Can be interrupted by timeout, any disconnect or fail, or by calling interruptWait()
         */
        bool waitForStateWithTimeout(WebsocketClient::State state, int timeoutMs);
        void interruptWait();

        /**
         * Set log channels for underlying websocket implementation.
         * Logs will be written only for channels that are enabled.
         * All channels not present in channelNames will be disabled.
         *
         * @param channelNames array of names of the channels to be set
         * @return true if channels were changed, false otherwise
         */
        bool setLogChannels(const Json::Value& channelNames);

    private:
        void connectAsyncImpl();
        void delayedReconnect();

        void onFail(std::weak_ptr<DisposableClient> wClientPtr, const Websocket::ConnectionInfo& connectionInfo);
        void onPong(std::weak_ptr<DisposableClient> wClientPtr, const Websocket::ConnectionInfo& info, const std::string& payload);
        void onClose(std::weak_ptr<DisposableClient> wClientPtr, const Websocket::ConnectionInfo& connectionInfo);
        void onOpen(std::weak_ptr<DisposableClient> wClientPtr);
        void onMessage(std::weak_ptr<DisposableClient> wClientPtr, const std::string& message);
        void onBinaryMessage(std::weak_ptr<DisposableClient> wClientPtr, const std::string& message);

        static websocketpp::log::level getBitMaskFromChannelNames(
            const Json::Value& channelNames, const std::map<std::string, websocketpp::log::level>& channelToInt);

        const std::shared_ptr<YandexIO::ITelemetry> telemetry_;
        websocketpp::log::level errorLogChannels_{};
        websocketpp::log::level accessLogChannels_{};

        OnConnect onConnect_;
        OnMessage onMessage_;
        OnMessage onBinaryMessage_;
        OnDisconnect onDisconnect_;
        OnFail onFail_;
        OnPong onPong_;

        std::atomic<int64_t> reconnectDelayMs_;

        bool stopped_ = false;       // Flag to stop all wait operations
        bool interruptWait_ = false; // Flag to interrupt waitForConnectedWithTimeout() before timeout

        /*
         * Is used only for stopped_ and interruptWait_
         */
        std::mutex mutex_;
        std::shared_ptr<DisposableClient> clientPtr_;

        Settings settings_;

        /*
         * state is can be changed only in in callbackQueue_ thread
         */
        std::atomic<State> state_{State::STOPPED};
        SteadyConditionVariable stateWakeUpVar_;
        /*
         * Must be destroyed the first
         */
        NamedCallbackQueue callbackQueue_{"WebsocketClient"};
    };

} // namespace quasar

bool operator==(const quasar::WebsocketClient::DisposableClient::Settings::TLS& settings_a,
                const quasar::WebsocketClient::DisposableClient::Settings::TLS& settings_b);

bool operator==(const quasar::WebsocketClient::DisposableClient::Settings& settings_a,
                const quasar::WebsocketClient::DisposableClient::Settings& settings_b);

bool operator==(const quasar::WebsocketClient::Settings& settings_a, const quasar::WebsocketClient::Settings& settings_b);
