#pragma once

#include <yandex_io/libs/logging/logging.h>

#include <websocketpp/logger/basic.hpp>
#include <websocketpp/logger/levels.hpp>

namespace quasar {

    /**
     * Logger for websocketpp that outputs to an ostream (stdout/stderr by default), to default Quasar logger (defined
     * in Logging.h)
     */
    template <typename concurrency, typename names>
    class WebsocketEventsLogger: public websocketpp::log::basic<concurrency, names> {
    private:
        using base_type = websocketpp::log::basic<concurrency, names>;
        using channel_type_hint = websocketpp::log::channel_type_hint;
        using alevel = websocketpp::log::alevel;
        using elevel = websocketpp::log::elevel;
        using level = websocketpp::log::level;

    public:
        WebsocketEventsLogger<concurrency, names>(channel_type_hint::value hint = channel_type_hint::access)
            : base_type(hint)
        {
        }

        WebsocketEventsLogger<concurrency, names>(std::ostream* out)
            : base_type(out)
        {
        }

        WebsocketEventsLogger<concurrency, names>(level channel, channel_type_hint::value hint = channel_type_hint::access)
            : base_type(channel, hint)
        {
        }

        WebsocketEventsLogger<concurrency, names>(level channel, std::ostream* out)
            : base_type(channel, out)
        {
        }

        WebsocketEventsLogger<concurrency, names>(const WebsocketEventsLogger<concurrency, names>& other)
            : base_type(other)
        {
        }

#ifdef _WEBSOCKETPP_DEFAULT_DELETE_FUNCTIONS_
        WebsocketEventsLogger<concurrency, names>& operator=(const WebsocketEventsLogger<concurrency, names>&) = delete;
#endif // _WEBSOCKETPP_DEFAULT_DELETE_FUNCTIONS_

#ifdef _WEBSOCKETPP_MOVE_SEMANTICS_
        WebsocketEventsLogger<concurrency, names>(WebsocketEventsLogger<concurrency, names>&& other) noexcept
            : base_type(std::move(other))
        {
        }

    #ifdef _WEBSOCKETPP_DEFAULT_DELETE_FUNCTIONS_
        WebsocketEventsLogger<concurrency, names>& operator=(WebsocketEventsLogger<concurrency, names>&&) = delete;
    #endif // _WEBSOCKETPP_DEFAULT_DELETE_FUNCTIONS_

#endif // _WEBSOCKETPP_MOVE_SEMANTICS_

        /**
         * Write a string message to the given channel
         * @param channel The channel to write to
         * @param msg The message to write
         */
        void write(level channel, const std::string& msg) {
            log(channel, msg);
            base_type::write(channel, msg);
        }

        /**
         * Write a cstring message to the given channel
         * @param channel The channel to write to
         * @param msg The message to write
         */
        void write(level channel, const char* msg) {
            std::string message(msg);
            log(channel, message);
            base_type::write(channel, msg);
        }

    private:
        void log(level channel, const std::string& msg) {
            YIO_LOG_TRACE("WebsocketEventsLogger::log channel=" << channel << ", msg=" << msg);
            if (this->dynamic_test(channel)) {
                YIO_LOG_INFO("[websocketpp] " << names::channel_name(channel) << ": " << msg);
            }
        }
    };

} // namespace quasar
