#pragma once

#include "quasar_ws_logger.h"
#include "websocket.h"

#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/threading/periodic_executor.h>

#include <websocketpp/server.hpp>
#include <websocketpp/config/asio.hpp>

#include <mutex>
#include <optional>
#include <set>
#include <string>

namespace websocketpp {
    namespace config {

        /// Server config with asio transport and TLS enabled
        /// Like stock `websocketpp::config::asio_tls` but with:
        /// -- custom logging
        struct quasar_ws_config: public core {
            using type = quasar_ws_config;
            using base = core;

            using concurrency_type = base::concurrency_type;

            using request_type = base::request_type;
            using response_type = base::response_type;

            using message_type = base::message_type;
            using con_msg_manager_type = base::con_msg_manager_type;
            using endpoint_msg_manager_type = base::endpoint_msg_manager_type;

            using alog_type = QuasarWsLogger<base::concurrency_type, log::alevel>;
            using elog_type = QuasarWsLogger<base::concurrency_type, log::elevel>;

            using rng_type = base::rng_type;

            struct transport_config: public base::transport_config {
                using concurrency_type = type::concurrency_type;
                using alog_type = type::alog_type;
                using elog_type = type::elog_type;
                using request_type = type::request_type;
                using response_type = type::response_type;
                using socket_type = websocketpp::transport::asio::tls_socket::endpoint;
            };

            using transport_type = websocketpp::transport::asio::endpoint<transport_config>;
        };
    } // namespace config
} // namespace websocketpp

namespace quasar {

    class WebsocketServer {
    public:
        using ConnectionHdl = websocketpp::connection_hdl;
        using OnMessage = std::function<void(WebsocketServer::ConnectionHdl hdl, const std::string&)>;
        using OnOpen = std::function<void(ConnectionHdl)>;
        using OnClose = std::function<void(ConnectionHdl, const Websocket::ConnectionInfo&)>;
        using OnPing = std::function<bool(websocketpp::connection_hdl hdl, const std::string& s)>;

        struct Settings {
            struct TLSSettings {
                bool disabled = false;
                std::string crtPemBuffer;
                std::string keyPemBuffer;
            };
            struct Ping {
                bool enabled = true;
                std::chrono::milliseconds interval = std::chrono::seconds(5);
            };
            struct Pong {
                bool enabled = true;
            };
            void ensureCorrect() const;
            Ping ping;
            Pong pong;
            TLSSettings tls;
            std::optional<int> port; // If not set any free port is used
            bool logErrorAsWarn = false;
        };

        WebsocketServer(Settings settings, std::shared_ptr<YandexIO::ITelemetry> telemetry);
        ~WebsocketServer();

        int start();
        void send(ConnectionHdl hdl, const std::string& msg);
        void sendAll(const std::string& msg);
        void close(ConnectionHdl hdl, const std::string& reason = "no reason",
                   Websocket::StatusCode code = Websocket::StatusCode::NORMAL);

        void setOnMessageHandler(OnMessage onMessage);
        void setOnOpenHandler(OnOpen onOpen);
        void setOnCloseHandler(OnClose onClose);
        /**
         * @brief Set custom ping handler. If pingHandler == nullptr -> default PingHandler will be set up
         */
        void setOnPingHandler(OnPing pingHandler);

        int getConnectionsNumber();

        /**
         * @brief Returns address/hostname of connection as string on success. Or empty string on failure.
         */
        std::string getRemoteHost(ConnectionHdl hdl);

        /**
         * @brief utility to convert string in form '[host]:port' to 'host'. Host may be strange like [::ffff:IPV4]
         */
        static std::string extractAsioHost(std::string /*src*/);

    private:
        using MessagePtr = websocketpp::config::asio::message_type::ptr;
        using ContextPtr = websocketpp::lib::shared_ptr<websocketpp::lib::asio::ssl::context>;

        static const std::string ciphers_;
        static const websocketpp::lib::asio::const_buffer dh_temp_params_;
        static const char dh_params_[];

        ContextPtr onTlsInit(ConnectionHdl hdl);
        void onMessage(ConnectionHdl hdl, MessagePtr msg);
        void onOpen(ConnectionHdl hdl);
        void onClose(ConnectionHdl hdl);
        void onPong(ConnectionHdl hdl, const std::string& payload);
        bool onPing(websocketpp::connection_hdl hdl, const std::string& payload);
        void run();
        void checkClientsLastPongs();
        void handleFatalException(const std::exception& exception, const std::string& msg);

        /*
         * Testing purposes
         */
        int findFirstFreePortAndListen();

        const std::shared_ptr<YandexIO::ITelemetry> telemetry_;

        OnMessage onMessage_;
        OnOpen onOpen_;
        OnClose onClose_;
        OnPing onPing_;

        std::mutex mutex_;
        websocketpp::server<websocketpp::config::quasar_ws_config> server_;
        std::thread serverThread_;
        Settings settings_;

        std::map<ConnectionHdl, std::chrono::steady_clock::time_point, std::owner_less<ConnectionHdl>> clientsLastPongs_;
        std::unique_ptr<PeriodicExecutor> lastPongCheckerPtr_;

        const double pingIntervalCoef_ = 2.1;
    };

} // namespace quasar
