#include "mini_unzip.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/logging/logging.h>

#include <contrib/libs/minizip/unzip.h>

#include <util/folder/path.h>
#include <util/generic/scope.h>

#include <fstream>
#include <string>
#include <vector>

#include <unistd.h>
#include <sys/stat.h>

YIO_DEFINE_LOG_MODULE("zip");

namespace quasar {

    namespace {

        int extractCurrentFile(unzFile uf, const std::string& destDir) {
            unz_file_info fileInfo;
            constexpr int MAX_FILE_SIZE = 1024;
            constexpr int READ_BUFFER_SIZE = 8092;
            std::vector<char> fileName(MAX_FILE_SIZE);
            std::vector<char> readBuffer(READ_BUFFER_SIZE);

            int err = unzGetCurrentFileInfo(uf, &fileInfo, &fileName[0], fileName.size(), nullptr, 0, nullptr, 0);

            if (err != UNZ_OK) {
                YIO_LOG_WARN("error " << err << " with zipfile in unzGetCurrentFileInfo");
                return err;
            }

            const auto destFile = JoinFsPaths(destDir, &fileName[0]);

            err = unzOpenCurrentFilePassword(uf, /*password*/ nullptr);
            if (err != UNZ_OK) {
                YIO_LOG_WARN("error " << err << " with zipfile in unzOpenCurrentFilePassword");
                return err;
            }

            Y_DEFER {
                unzCloseCurrentFile(uf);
            };

            // Biggest byte of version shows platform format of external attributes
            // 3 is a constant for UNIX in .ZIP specification
            // (look to https://pkware.cachefly.net/webdocs/casestudies/APPNOTE.TXT paragraph 4.4.2)
            if ((fileInfo.version >> 8) != 3) {
                YIO_LOG_ERROR_EVENT("Zip.ArchiveNotUnixCompatible", "Zip archive is not UNIX-compatible");
                return -1;
            }
            // Biggest 2 bytes contain info about file type and permissions in UNIX format
            auto fileType = fileInfo.external_fa >> 16;
            TFsPath(destFile).Parent().MkDirs();

            if ((fileType & S_IFLNK) == S_IFLNK) {
                // Is symlink
                int bytesRead = unzReadCurrentFile(uf, readBuffer.data(), readBuffer.size() - 1);
                readBuffer[bytesRead] = '\0';
                if (int errCode = symlink(readBuffer.data(), destFile.c_str())) {
                    YIO_LOG_ERROR_EVENT("Zip.CreateSymlinkFail", "Can't create symlink, error is " << strError(errno));
                    return errCode;
                }
                return 0;
            }
            if ((fileType & S_IFDIR) == S_IFDIR) {
                // It's directory, we don't need to write anything
                return 0;
            }
            std::ofstream outFile(destFile, std::ios::binary);
            if (!outFile.good()) {
                YIO_LOG_ERROR_EVENT("Zip.CreateOutFileFail", "Can't create " << destFile);
                return -1;
            }
            while (true) {
                int bytesWritten = unzReadCurrentFile(uf, &readBuffer[0], readBuffer.size());
                if (bytesWritten < 0) {
                    YIO_LOG_WARN("error " << bytesWritten << " with zipfile in unzReadCurrentFile");
                    return bytesWritten;
                } else if (bytesWritten > 0) {
                    outFile.write(&readBuffer[0], bytesWritten);
                } else {
                    break;
                }
            }
            return 0;
        }

    } // namespace

    bool extractArchiveMinizip(const std::string& archFileName, const std::string& destDir) {
        unzFile zipArch = unzOpen(archFileName.c_str());

        Y_DEFER {
            unzClose(zipArch);
        };

        unz_global_info gi;

        int err = unzGetGlobalInfo(zipArch, &gi);

        if (err != UNZ_OK) {
            YIO_LOG_WARN("error " << err << " with zipfile in unzGetGlobalInfo");
            return false;
        }

        for (size_t i = 0; i < gi.number_entry; ++i)
        {
            if (extractCurrentFile(zipArch, destDir) != UNZ_OK) {
                return false;
            }

            if ((i + 1) < gi.number_entry) {
                err = unzGoToNextFile(zipArch);
                if (err != UNZ_OK) {
                    YIO_LOG_WARN("error " << err << " with zipfile in unzGoToNextFile");
                    return false;
                }
            }
        }

        return true;
    }

    void extractZipArchive(const std::string& archivePath, const std::string& destPath) {
        TFsPath(destPath).MkDirs();
        if (!extractArchiveMinizip(archivePath, destPath)) {
            throw std::runtime_error("Failed to extract spotter archive " + archivePath + " to " + destPath);
        }
    }

} // namespace quasar
