#include "metrica_endpoint.h"

#include "metrica2_service.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

YIO_DEFINE_LOG_MODULE("clickdaemon_metrica");

namespace quasar {

    using quasar::proto::QuasarMessage;

    const std::string MetricaEndpoint::SERVICE_NAME = "metrica2d";

    MetricaEndpoint::MetricaEndpoint(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory)
        : device_(std::move(device))
        , server_(ipcFactory->createIpcServer(MetricaEndpoint::SERVICE_NAME))
        , configHelper_(device_->configuration()->getServiceConfig(MetricaEndpoint::SERVICE_NAME))
        , clientStats_([this](Json::Value value) {
            for (auto& consumer : consumers_) {
                consumer->processStats(value);
            }
        })
    {
        server_->setMessageHandler(std::bind(&MetricaEndpoint::processQuasarMessage, this, std::placeholders::_1, std::placeholders::_2));
    }

    int MetricaEndpoint::port() const {
        return server_->port();
    }

    void MetricaEndpoint::start() {
        server_->listenService();
    }

    MetricaEndpoint::~MetricaEndpoint() {
        server_->shutdown();
    }

    void MetricaEndpoint::processQuasarMessage(const ipc::SharedMessage& message, ipc::IServer::IClientConnection& connection) {
        if (message->has_metrica_message()) {
            YIO_LOG_DEBUG("Got metrica message: " << shortUtf8DebugString(*message));

            const auto& metricaMessage = message->metrica_message();
            const YandexIO::ITelemetry::Flags flags = metricaMessage.flags();
            const bool skipDatabase = flags & YandexIO::ITelemetry::SKIP_DATABASE;

            if (metricaMessage.has_report_event()) {
                const auto& eventName = metricaMessage.report_event();
                const auto& eventMessage = metricaMessage.has_report_event_json_value()
                                               ? metricaMessage.report_event_json_value()
                                               : "";
                for (auto& consumer : consumers_) {
                    consumer->processEvent(eventName, eventMessage, skipDatabase);
                }
            }

            if (metricaMessage.has_report_error()) {
                const auto& errorEventName = metricaMessage.report_error();
                const auto& reportErrorValue = metricaMessage.report_error_value();

                for (auto& consumer : consumers_) {
                    consumer->processError(errorEventName, reportErrorValue, skipDatabase);
                }
            }

            if (metricaMessage.has_report_key_value()) {
                const auto& report = metricaMessage.report_key_value();
                const auto& eventName = report.event_name();

                auto makeEventValue = [&report]() -> std::string {
                    Json::Value json;
                    for (const auto& kv : report.key_values()) {
                        json[kv.first] = kv.second;
                    }
                    return jsonToString(json);
                };

                const auto eventValue = makeEventValue();

                for (auto& consumer : consumers_) {
                    consumer->processEvent(eventName, eventValue, skipDatabase);
                }
            }

            if (metricaMessage.has_app_environment_value()) {
                for (auto& consumer : consumers_) {
                    consumer->putEnvironmentVariable(
                        metricaMessage.app_environment_value().key(),
                        metricaMessage.app_environment_value().value());
                }
            }

            if (metricaMessage.has_delete_environment_key()) {
                for (auto& consumer : consumers_) {
                    consumer->deleteEnvironmentVariable(metricaMessage.delete_environment_key());
                }
            }

            if (metricaMessage.has_create_latency_point()) {
                latencyPointsStart_[metricaMessage.create_latency_point()] = std::chrono::steady_clock::now();
            }

            if (metricaMessage.has_report_latency_event()) {
                auto now = std::chrono::steady_clock::now();
                const std::string latencyPointName = metricaMessage.report_latency_event().latency_point();
                auto it = latencyPointsStart_.find(latencyPointName);
                if (it != latencyPointsStart_.end()) {
                    Json::Value json;
                    if (metricaMessage.has_report_event_json_value()) {
                        const auto& eventMessage = metricaMessage.report_event_json_value();
                        auto parsed = tryParseJson(eventMessage);
                        if (parsed) {
                            json = std::move(*parsed);
                        }
                    }
                    json["value"] = static_cast<int64_t>(std::chrono::duration_cast<std::chrono::milliseconds>(now - it->second).count());
                    const auto latencyValue = jsonToString(json);

                    for (auto& consumer : consumers_) {
                        consumer->processEvent(metricaMessage.report_latency_event().event_name(), latencyValue);
                    }

                    if (metricaMessage.report_latency_event().has_remove_point() && metricaMessage.report_latency_event().remove_point()) {
                        latencyPointsStart_.erase(it);
                    }
                } else {
                    if (!metricaMessage.report_latency_event().silent_if_not_exists()) {
                        YIO_LOG_INFO("Latency point \"" << latencyPointName << "\" not found but message to report it was received");
                    }
                }
            }

            if (metricaMessage.has_network_status()) {
                const auto& networkStatus = metricaMessage.network_status();

                for (auto& consumer : consumers_) {
                    consumer->setConnectionType(networkStatus.type());
                }
            }

            if (metricaMessage.has_config()) {
                auto systemConfig = parseJson(metricaMessage.config())["system_config"];
                processMetricaConfig(tryGetJson(systemConfig, "metricad"));
            }

            if (metricaMessage.has_stats()) {
                clientStats_.processStatsMessage(metricaMessage.stats());
            }

            if (message->has_request_id()) {
                // Send acknowledge
                proto::QuasarMessage response;
                response.set_request_id(message->request_id());
                connection.send(std::move(response));
            }
        }
    }

    void MetricaEndpoint::processMetricaConfig(const Json::Value& config) {
        auto configUpdate = configHelper_.getConfigurationUpdate(config);
        if (configUpdate.isNull()) {
            return;
        }

        YIO_LOG_INFO("Got metrica config update: " << jsonToString(configUpdate));

        auto currentConfig = configHelper_.getCurrentConfig();
        auto consumerConfigUpdates = tryGetJson(configUpdate, "consumers");
        for (auto& consumer : consumers_) {
            auto consumerConfigUpdate = tryGetJson(consumerConfigUpdates, consumer->getName());
            auto consumerConfig = tryGetJson(currentConfig["consumers"], consumer->getName());

            if (consumerConfigUpdate.isObject()) {
                consumer->processConfigUpdate(consumerConfigUpdate, consumerConfig);
            }
        }

        clientStats_.applyConfig(currentConfig);
    }

} /* namespace quasar */
