#include "app_metrica_endpoint.h"

#include "metrica_service.h"

#include <yandex_io/libs/appmetrica/app_metrica.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/metrica/base/events_database_creator.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

#include <google/protobuf/text_format.h>

#include <chrono>
#include <ctime>
#include <fstream>
#include <memory>
#include <string>

YIO_DEFINE_LOG_MODULE("appmetrica");

using namespace std::chrono;

namespace quasar {

    using quasar::proto::QuasarMessage;
    using quasar::proto::WifiList;
    using std::string;

    const std::string AppMetricaEndpoint::SERVICE_NAME = "metricad";

    /**
     * @brief Convert quasar protobuf WifiList message to Metrica protobuf NetworkInfo message.
     *        Copy wifi networks and set up ConnectionType to WIFI
     * @param wifiList - scanned wifi networks
     * @return Metrica NetworkInfo message with wifi_networks and connection_type set up
     */
    static ReportMessage_Session_Event_NetworkInfo convertWifiList(const WifiList& wifiList);

    AppMetricaEndpoint::AppMetricaEndpoint(
        std::shared_ptr<YandexIO::IDevice> device,
        const std::shared_ptr<ipc::IIpcFactory>& ipcFactory,
        std::shared_ptr<ReportConfiguration> reportConfig,
        std::chrono::milliseconds reportSendPeriod)
        : device_(std::move(device))
        , reportConfig_(reportConfig)
        , clientStats_([this](Json::Value event) {
            appMetrica_->processStats(std::move(event));
        })
    {
        auto serviceConfig = device_->configuration()->getServiceConfig(SERVICE_NAME);
        std::string sessionIdPersistentPart = getString(serviceConfig, "metricaSessionIdPersistentPart");
        std::string sessionIdTemporaryPart = getString(serviceConfig, "metricaSessionIdTemporaryPart");

        const auto& commonConfig = device_->configuration()->getServiceConfig("common");
        auto eventsDatabase = createEventsDatabase(commonConfig["eventsDatabase"]);
        appMetrica_ = std::make_unique<AppMetrica>(reportSendPeriod, reportConfig, std::move(eventsDatabase),
                                                   std::move(sessionIdPersistentPart), std::move(sessionIdTemporaryPart), device_);

        configHelper_ = std::make_unique<ConfigurationHelper>(serviceConfig);

        initIpc(ipcFactory);

        if (serviceConfig.isMember("appmetrica")) {
            processAppmetricaConfig(serviceConfig["appmetrica"]);
        }
    }

    AppMetricaEndpoint::AppMetricaEndpoint(
        std::shared_ptr<YandexIO::IDevice> device,
        const std::shared_ptr<ipc::IIpcFactory>& ipcFactory)
        : device_(std::move(device))
        , clientStats_([this](Json::Value event) {
            appMetrica_->processStats(std::move(event));
        })
    {
        auto serviceConfig = device_->configuration()->getServiceConfig(SERVICE_NAME);
        std::string sessionIdPersistentPart = getString(serviceConfig, "metricaSessionIdPersistentPart");
        std::string sessionIdTemporaryPart = getString(serviceConfig, "metricaSessionIdTemporaryPart");

        auto commonConfig = device_->configuration()->getServiceConfig("common");
        auto eventsDatabase = createEventsDatabase(commonConfig["eventsDatabase"]);

        reportConfig_ = std::make_shared<ReportConfiguration>(std::time(nullptr), std::vector<std::string>(), *device_);
        appMetrica_ = std::make_unique<AppMetrica>(std::move(eventsDatabase), std::move(sessionIdPersistentPart), std::move(sessionIdTemporaryPart), device_);

        configHelper_ = std::make_unique<ConfigurationHelper>(serviceConfig);

        initIpc(ipcFactory);

        if (serviceConfig.isMember("appmetrica")) {
            processAppmetricaConfig(serviceConfig["appmetrica"]);
        }

        startInitConfigThread();

        if (tryGetBool(serviceConfig, "metrica2Enabled", false)) {
            onMetrica2Done_ = [](const auto& /*response*/) {};
            onMetrica2Error_ = [](const std::string& error) {
                YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.ForwardMsgToMetrica2dError", "Error sending message to metrica2d: " << error);
            };

            metrica2Connector_->connectToService();
            if (metrica2Connector_->waitUntilConnected(std::chrono::seconds(10))) {
                YIO_LOG_INFO("metrica2d connected");
            } else {
                YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.ConnectToMetrica2.Timeout", "Could not connect to metrica2d in 10 seconds");
            }
        }
    }

    void AppMetricaEndpoint::initIpc(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory) {
        metrica2Connector_ = ipcFactory->createIpcConnector("metrica2d");
        metrica2Connector_->setConnectHandler([this] {
            std::scoped_lock lock(configMutex_);
            if (configCache_.empty()) {
                return;
            }
            proto::QuasarMessage message;
            message.mutable_metrica_message()->set_config(TString(configCache_));
            metrica2Connector_->sendMessage(std::move(message));
        });

        server_ = ipcFactory->createIpcServer(AppMetricaEndpoint::SERVICE_NAME);
        server_->setMessageHandler(std::bind(&AppMetricaEndpoint::processQuasarMessage, this, std::placeholders::_1, std::placeholders::_2));
        server_->setClientConnectedHandler(std::bind(&AppMetricaEndpoint::onClientConnected, this, std::placeholders::_1));
        server_->listenService();
    }

    int AppMetricaEndpoint::port() const {
        return server_->port();
    }

    void AppMetricaEndpoint::setPassportUid(const std::string& uid) {
        if (passportUid_.value_or("") != uid) {
            passportUid_ = uid;
            appMetrica_->metricaClient->setUserInfo(uid);
            appMetrica_->metricaClient->reportEvent("passportUidReceived", "");
        }
    }

    void AppMetricaEndpoint::onClientConnected(ipc::IServer::IClientConnection& connection) {
        std::lock_guard<std::mutex> lock(startupIdentifiersMutex_);
        if (!metricaUuid_.empty()) {
            YIO_LOG_INFO("Client connected, sending UUID");

            quasar::proto::QuasarMessage message;
            if (!metricaUuid_.empty()) {
                message.mutable_metrica_message()->set_metrica_uuid(TString(metricaUuid_));
            }
            connection.send(std::move(message));
        } else {
            YIO_LOG_INFO("Client connected, no UUID yet");
        }
    }

    void AppMetricaEndpoint::setUuid(const std::string& uuid) {
        std::lock_guard<std::mutex> lock(startupIdentifiersMutex_);
        metricaUuid_ = uuid;

        YIO_LOG_INFO("Got UUID, sending to all clients");

        quasar::proto::QuasarMessage message;
        message.mutable_metrica_message()->set_metrica_uuid(TString(metricaUuid_));
        server_->sendToAll(std::move(message));
    }

    AppMetricaEndpoint::~AppMetricaEndpoint() {
        server_->shutdown();
        if (initConfigThread_.joinable()) {
            initConfigThread_.join();
        }
    }

    void AppMetricaEndpoint::processQuasarMessage(const ipc::SharedMessage& message, ipc::IServer::IClientConnection& connection) {
        {
            /* Restart session each hour, so daily location charts will work (it needs the session start) */
            std::lock_guard<std::mutex> guard(startSessionMutex_);
            const auto now = steady_clock::now();
            if (duration_cast<minutes>(now - lastStartedSessionTimePoint_).count() > 60) {
                lastStartedSessionTimePoint_ = now;
                appMetrica_->metricaClient->startSession();
            }
        }

        if (message->has_metrica_message()) {
            YIO_LOG_DEBUG("Got metrica message: " << shortUtf8DebugString(*message));

            if (metrica2Connector_->isConnected()) {
                QuasarMessage metrica2Message(*message);
                metrica2Connector_->sendRequest(std::move(metrica2Message), onMetrica2Done_, onMetrica2Error_, std::chrono::seconds(5));
            }

            const auto& metricaMessage = message->metrica_message();

            const YandexIO::ITelemetry::Flags flags = metricaMessage.flags();

            const bool skipDatabase = flags & YandexIO::ITelemetry::SKIP_DATABASE;

            if (metricaMessage.has_report_event()) {
                const auto& eventName = metricaMessage.report_event();
                if (metricaMessage.has_report_event_json_value()) {
                    appMetrica_->metricaClient->reportEvent(eventName, metricaMessage.report_event_json_value(), skipDatabase);
                } else {
                    appMetrica_->metricaClient->reportEvent(eventName, std::string(), skipDatabase);
                }
            }

            if (metricaMessage.has_report_error()) {
                const auto& errorEventName = metricaMessage.report_error();
                const auto& reportErrorValue = metricaMessage.report_error_value();
                appMetrica_->metricaClient->reportError(errorEventName, skipDatabase, reportErrorValue);
            }

            if (metricaMessage.has_report_key_value()) {
                const auto& report = metricaMessage.report_key_value();
                const auto& eventName = report.event_name();
                auto makePayload = [&report]() {
                    Json::Value json;
                    for (const auto& kv : report.key_values()) {
                        json[kv.first] = kv.second;
                    }
                    return jsonToString(json);
                };
                appMetrica_->metricaClient->reportEvent(eventName, makePayload(), skipDatabase);
            }

            if (metricaMessage.has_app_environment_value()) {
                appMetrica_->metricaClient->putEnvironmentVariable(
                    metricaMessage.app_environment_value().key(),
                    metricaMessage.app_environment_value().value());
            }

            if (metricaMessage.has_delete_environment_key()) {
                appMetrica_->metricaClient->deleteEnvironmentVariable(metricaMessage.delete_environment_key());
            }

            if (metricaMessage.has_create_latency_point()) {
                latencyPointsStart_[metricaMessage.create_latency_point()] = std::chrono::steady_clock::now();
            }

            if (metricaMessage.has_report_latency_event()) {
                auto now = std::chrono::steady_clock::now();
                string latencyPointName = metricaMessage.report_latency_event().latency_point();
                auto it = latencyPointsStart_.find(latencyPointName);
                if (it != latencyPointsStart_.end()) {
                    Json::Value json;
                    if (metricaMessage.has_report_event_json_value()) {
                        const auto& eventMessage = metricaMessage.report_event_json_value();
                        auto parsed = tryParseJson(eventMessage);
                        if (parsed) {
                            json = std::move(*parsed);
                        }
                    }
                    json["value"] = static_cast<int64_t>(std::chrono::duration_cast<std::chrono::milliseconds>(now - it->second).count());
                    Json::FastWriter wr;
                    appMetrica_->metricaClient->reportEvent(metricaMessage.report_latency_event().event_name(), wr.write(json));
                    if (metricaMessage.report_latency_event().has_remove_point() && metricaMessage.report_latency_event().remove_point()) {
                        latencyPointsStart_.erase(it);
                    }

                    if (YIO_LOG_DEBUG_ENABLED()) {
                        const auto eventName = metricaMessage.report_latency_event().event_name();
                        YIO_LOG_DEBUG("Measured latency for " << eventName << ": " << json["value"]);
                    }
                } else {
                    if (!metricaMessage.report_latency_event().silent_if_not_exists()) {
                        YIO_LOG_INFO("Latency point \"" << latencyPointName << "\" not found but message to report it was received");
                    }
                }
            }

            if (metricaMessage.has_network_status()) {
                appMetrica_->metricaClient->setConnectionType(metricaMessage.network_status().type());
            }

            if (metricaMessage.has_wifi_list()) {
                if (appMetrica_->metricaSenderInitialized()) {
                    appMetrica_->metricaSender->setNetworkInfo(convertWifiList(metricaMessage.wifi_list()));
                }
            }

            if (metricaMessage.has_location()) {
                const auto& location = metricaMessage.location();

                /* Set up location for whole ReportMessage session */
                if (appMetrica_->metricaSenderInitialized()) {
                    appMetrica_->metricaSender->setLocation(location.latitude(), location.longitude());
                } else {
                    /* appMetrica_->metricaSender does not exists yet, so save location and set up it when sender will be created */
                    locationCache_.CopyFrom(location);
                }
            }

            if (metricaMessage.has_timezone()) {
                const auto& timezone = metricaMessage.timezone();

                if (timezone.has_timezone_name()) {
                    /* Set up environment value with timezone name */
                    appMetrica_->metricaClient->putEnvironmentVariable("timezone", timezone.timezone_name());
                }
                if (timezone.has_timezone_offset_sec()) {
                    /* Set up timezone offset for whole ReportMessage session */
                    if (appMetrica_->metricaSenderInitialized()) {
                        appMetrica_->metricaSender->setTimezoneOffsetSec(timezone.timezone_offset_sec());
                    } else {
                        /* appMetrica_->metricaSender does not exists yet, save timezone offset and set up when it will be created */
                        timezoneCache_.CopyFrom(timezone);
                    }
                }
            }

            if (metricaMessage.has_config()) {
                const auto systemConfig = parseJson(metricaMessage.config())["system_config"];
                const auto& appmetricaConfig = systemConfig["metricad"]["consumers"]["appmetrica"];

                {
                    std::scoped_lock lock(configMutex_);
                    configCache_ = metricaMessage.config();
                }

                processAppmetricaConfig(appmetricaConfig);
                metrica2Connector_->sendMessage(message);
            }

            if (metricaMessage.has_stats()) {
                clientStats_.processStatsMessage(metricaMessage.stats());
            }

            if (message->has_request_id()) {
                // Send acknowledge
                quasar::proto::QuasarMessage response;
                response.set_request_id(message->request_id());
                connection.send(std::move(response));
            }

            if (afterHandleMetricaMessage) {
                afterHandleMetricaMessage(metricaMessage);
            }
        }
    }

    void AppMetricaEndpoint::processAppmetricaConfig(const Json::Value& appmetricaConfig) {
        const auto configUpdate = configHelper_->getConfigurationUpdate(appmetricaConfig);
        const auto currentConfig = configHelper_->getCurrentConfig();

        if (configUpdate.isMember("eventBlacklist")) {
            appMetrica_->metricaClient->updateEventBlacklist(currentConfig["eventBlacklist"]);
        }

        if (configUpdate.isMember("eventImportantList")) {
            auto eventImportantList = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(currentConfig, "eventImportantList", {});
            auto importantEvents = std::make_shared<decltype(eventImportantList)>(std::move(eventImportantList));
            YIO_LOG_DEBUG("AppMetrica eventImportantList: [" << join(*importantEvents, ",") << "]");
            importantEvents_ = (std::lock_guard(importantEventsMutex_), importantEvents);

            if (appMetrica_->metricaSenderInitialized()) {
                appMetrica_->metricaSender->setEventImportantList(std::move(importantEvents));
            }
        }

        if (configUpdate.isMember("envKeysBlacklist")) {
            auto envKeysBlacklist = tryGetEmplaceableStringSet<std::unordered_set<std::string>>(
                currentConfig, "envKeysBlacklist", {});
            if (appMetrica_->metricaSenderInitialized()) {
                appMetrica_->metricaSender->setEnvKeysBlacklist(std::move(envKeysBlacklist));
            } else {
                YIO_LOG_DEBUG("cached envKeysBlacklist");
                envKeysBlacklistCache_ = std::move(envKeysBlacklist);
            }
        }

        clientStats_.applyConfig(currentConfig);
    }

    std::shared_ptr<EventsDatabase> AppMetricaEndpoint::createEventsDatabase(const Json::Value& config) {
        return quasar::createEventsDatabase(config,
                                            [this](uint64_t /*id*/, const proto::DatabaseMetricaEvent& event, bool saved) -> std::optional<bool> {
                                                if (!saved && event.has_new_event()) {
                                                    auto importantEvents = (std::lock_guard(importantEventsMutex_), importantEvents_);
                                                    if (importantEvents && importantEvents->count(event.new_event().name())) {
                                                        return true;
                                                    }
                                                }
                                                return std::nullopt;
                                            });
    }

    void AppMetricaEndpoint::startInitConfigThread() {
        initConfigThread_ = std::thread(&AppMetricaEndpoint::initConfigAndStart, this);
    }

    void AppMetricaEndpoint::initConfigAndStart() {
        const auto& serviceConfig = device_->configuration()->getServiceConfig(AppMetricaEndpoint::SERVICE_NAME);

        const std::string appMetricaStartupHost = serviceConfig["appMetricaStartupHost"].asString();
        const std::string appMetricaApiKey = serviceConfig["apiKey"].asString();
        const std::string metricaMetadataPath = serviceConfig.isMember("metricaMetadataPath")
                                                    ? serviceConfig["metricaMetadataPath"].asString()
                                                    : DEFAULT_METRICA_METADATA_PATH;

        const Json::Value commonConfig = device_->configuration()->getServiceConfig("common");
        const std::string deviceType = commonConfig["deviceType"].asString();
        const std::string metricaAppId = commonConfig["metricaAppId"].asString();

        auto metricaMetadata = loadMetadata(metricaMetadataPath);
        const bool uuidSet = !metricaMetadata.UUID.empty();

        metricaMetadata.deviceID = device_->deviceId();

        if (uuidSet) {
            YIO_LOG_INFO("Using cached UUID");
            setUuid(metricaMetadata.UUID);
        }

        const auto appVersionName = device_->softwareVersion();
        auto appVersion = appVersionName;
        appVersion.erase(std::remove_if(appVersion.begin(), appVersion.end(),
                                        [](char c) { return !std::isdigit(c); }), appVersion.end());

        auto startupConfig = StartupConfiguration{
            .startTime = std::time(nullptr),
            .startupHost = appMetricaStartupHost,
            .apiKey = appMetricaApiKey,
            .deviceID = metricaMetadata.deviceID,
            .UUID = metricaMetadata.UUID,
            .model = deviceType,
            .appVersion = appVersion,
            .appVersionName = appVersionName,
            .metricaAppID = metricaAppId,
            .revision = device_->platformRevision(),
            .osVersion = StartupConfiguration::systemOsVersion(),
        };

        metricaMetadata = appMetrica_->initMetricaHttpClient(std::move(startupConfig));

        if (!uuidSet) {
            saveMetadata(metricaMetadataPath, metricaMetadata);

            YIO_LOG_INFO("Using new UUID");
            setUuid(metricaMetadata.UUID);
        }

        if (appMetrica_->metricaSenderInitialized()) {
            /* Set up cached location and timezone if have it */
            if (timezoneCache_.has_timezone_offset_sec()) {
                appMetrica_->metricaSender->setTimezoneOffsetSec(timezoneCache_.timezone_offset_sec());
            }
            if (locationCache_.has_longitude() && locationCache_.has_latitude()) {
                appMetrica_->metricaSender->setLocation(locationCache_.latitude(), locationCache_.longitude());
            }

            auto importantEvents = (std::lock_guard(importantEventsMutex_), importantEvents_);
            if (importantEvents && !importantEvents->empty()) {
                YIO_LOG_DEBUG("got eventImportantList from cache");
                appMetrica_->metricaSender->setEventImportantList(importantEvents);
            }
            if (!envKeysBlacklistCache_.empty()) {
                YIO_LOG_DEBUG("got envKeysBlacklist from cache");
                appMetrica_->metricaSender->setEnvKeysBlacklist(std::move(envKeysBlacklistCache_));
            }
        }
    }

    MetricaMetadata AppMetricaEndpoint::loadMetadata(const std::string& filepath)
    {
        if (!fileExists(filepath)) {
            return MetricaMetadata{};
        }

        std::ifstream file{filepath};
        if (!file.is_open()) {
            YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.LoadMetadata.OpenFailed", "Failed to open " << filepath);
            return MetricaMetadata{};
        }

        try {
            return MetricaMetadata::fromJson(parseJson(file));
        } catch (const Json::Exception& e) {
            YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.LoadMetadata.InvalidJson", "Failed to read from " << filepath << ", JSON is broken (" << e.what() << ")");
        } catch (const std::logic_error& e) {
            YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.LoadMetadata.ReadJsonFailed", "Failed to read from JSON. " << e.what());
        }

        return MetricaMetadata{};
    }

    void AppMetricaEndpoint::saveMetadata(const std::string& filepath, const MetricaMetadata& metadata)
    {
        const auto json{metadata.toJson()};

        TransactionFile file{filepath};
        try {
            if (!file.write(jsonToString(json))) {
                YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.SaveMetadata.WriteFailed", "Failed to write to temporary file.");
                return;
            }
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.SaveMetadata.WriteException", "Failed to write to temporary file. " << e.what());
            return;
        }

        if (!file.commit()) {
            YIO_LOG_ERROR_EVENT("AppMetricaEndpoint.SaveMetadata.CommitFailed", "Failed to write to persistent file.");
        }
    }

    ReportMessage_Session_Event_NetworkInfo convertWifiList(const WifiList& wifiList)
    {
        ReportMessage_Session_Event_NetworkInfo networkInfo;

        for (const auto& hotspot : wifiList.hotspots()) {
            auto network = networkInfo.add_wifi_networks();
            network->set_mac(hotspot.mac());
            network->set_signal_strength(hotspot.rssi());
            network->set_ssid(hotspot.ssid());
            network->set_is_connected(hotspot.is_connected());
        }
        return networkInfo;
    }

} /* namespace quasar */
