#include "metrica_connector.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/rate_limiter/bucket/factory.h>
#include <yandex_io/protos/quasar_proto.pb.h>

YIO_DEFINE_LOG_MODULE("metrica");

using namespace quasar;
using namespace quasar::proto;

namespace {

    constexpr size_t DEFAULT_MESSAGES_TO_SEND = 100;

    ConnectionType toProtoConnectionType(MetricaConnector::Params::ConnectionType type) {
        switch (type) {
            case MetricaConnector::Params::ConnectionType::UNKNOWN:
                return CONNECTION_TYPE_UNKNOWN;
            case MetricaConnector::Params::ConnectionType::WIFI:
                return CONNECTION_TYPE_WIFI;
            case MetricaConnector::Params::ConnectionType::ETHERNET:
                return CONNECTION_TYPE_ETHERNET;
        }
    }

    NetworkStatus toProtoNetworkStatus(MetricaConnector::Params::NetworkStatus status) {
        auto toProtoStatus = [](MetricaConnector::Params::NetworkStatus::Status status) {
            switch (status) {
                case MetricaConnector::Params::NetworkStatus::NOT_CONNECTED:
                    return NetworkStatus::NOT_CONNECTED;
                case MetricaConnector::Params::NetworkStatus::CONNECTING:
                    return NetworkStatus::CONNECTING;
                case MetricaConnector::Params::NetworkStatus::CONNECTED_NO_INTERNET:
                    return NetworkStatus::CONNECTED_NO_INTERNET;
                case MetricaConnector::Params::NetworkStatus::CONNECTED:
                    return NetworkStatus::CONNECTED;
                case MetricaConnector::Params::NetworkStatus::NOT_CHOSEN:
                    return NetworkStatus::NOT_CHOSEN;
            }
        };

        NetworkStatus networkStatus;
        networkStatus.set_status(toProtoStatus(status.status));
        networkStatus.set_type(toProtoConnectionType(status.type));
        return networkStatus;
    }

    bool isEventMessage(const MetricaMessage& message) {
        return message.has_report_latency_event() || message.has_report_event() || message.has_report_error();
    }

    template <typename Dest, typename Src, typename Enum>
    void addCounter(Dest& dst, const Src& counters, Enum id, const std::string& name) {
        auto data = dst->add_data();
        data->set_name(TString(name));
        for (unsigned i = 0; i < counters.size(); ++i) {
            data->add_counters(counters[i].get()[id]);
        }
    }
} // namespace

MetricaConnector::MetricaConnector(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory, const std::string& serviceName)
    : messagesToSend_(DEFAULT_MESSAGES_TO_SEND)
    , params_(std::make_unique<Params>(this))
{
    connector_ = ipcFactory->createIpcConnector(serviceName);
    connector_->setConnectHandler(std::bind(&MetricaConnector::handleReconnect, this));
    connector_->setMessageHandler(std::bind(&MetricaConnector::handleQuasarMessage, this, std::placeholders::_1));

    senderThread_ = std::thread(&MetricaConnector::senderThread, this);
    connector_->connectToService();

    periodicExecutor_ = std::make_unique<PeriodicExecutor>(
        [this]() {
            QuasarMessage message;
            auto statsMsg = message.mutable_metrica_message()->mutable_stats();
            statsMsg->set_service_name(TString(senderName_));
            const auto counters = dailyStats_.getUpdatedCounters();
            addCounter(statsMsg, counters, Counters::TOTAL, "total");
            addCounter(statsMsg, counters, Counters::QUEUE_DROP, "queueDrop");
            addCounter(statsMsg, counters, Counters::RATELIMIT_DROP, "ratelimitDrop");
            enqueue(message);
        },
        std::chrono::minutes(5));
}

void MetricaConnector::handleReconnect()
{
    YIO_LOG_INFO("Handle reconnect");
    {
        std::scoped_lock guard(environmentVariablesMutex_);
        for (const auto& var : environmentVariables_) {
            YIO_LOG_INFO("Handle reconnect: " << var.first);
            QuasarMessage message;
            message.mutable_metrica_message()->mutable_app_environment_value()->set_key(TString(var.first));
            message.mutable_metrica_message()->mutable_app_environment_value()->set_value(TString(var.second));
            enqueue(message);
        }
    }
    senderCV_.notify_all();
    isOnConnectedCalled_.store(true);
}

void MetricaConnector::handleQuasarMessage(const ipc::SharedMessage& message) {
    if (message->has_metrica_message() && message->metrica_message().has_metrica_uuid()) {
        std::scoped_lock lock(uuidMutex_);

        uuid_ = message->metrica_message().metrica_uuid();

        for (auto cb : uuidCallbacks_) {
            cb(*uuid_);
        }

        uuidCallbacks_.clear();
    }
}

void MetricaConnector::reportEvent(const std::string& event, ITelemetry::Flags flags)
{
    if (isOverflowed(event)) {
        return;
    }

    QuasarMessage message;
    message.mutable_metrica_message()->set_flags(flags);
    message.mutable_metrica_message()->set_report_event(TString(event));
    enqueue(message);
}

void MetricaConnector::reportEvent(const std::string& event, const std::string& eventJson, ITelemetry::Flags flags)
{
    if (isOverflowed(event)) {
        return;
    }

    QuasarMessage message;
    message.mutable_metrica_message()->set_flags(flags);
    message.mutable_metrica_message()->set_report_event(TString(event));
    message.mutable_metrica_message()->set_report_event_json_value(TString(eventJson));
    enqueue(message);
}

void MetricaConnector::reportError(const std::string& errorEventName, ITelemetry::Flags flags)
{
    if (isOverflowed(errorEventName)) {
        return;
    }

    QuasarMessage message;
    message.mutable_metrica_message()->set_flags(flags);
    message.mutable_metrica_message()->set_report_error(TString(errorEventName));
    enqueue(message);
}

void MetricaConnector::reportError(const std::string& errorEventName, const std::string& errorValue, ITelemetry::Flags flags)
{
    if (isOverflowed(errorEventName)) {
        return;
    }

    QuasarMessage message;
    message.mutable_metrica_message()->set_flags(flags);
    message.mutable_metrica_message()->set_report_error(TString(errorEventName));
    message.mutable_metrica_message()->set_report_error_value(TString(errorValue));
    enqueue(message);
}

void MetricaConnector::reportLogError(const std::string& message, const std::string& sourceFileName, size_t sourceLine, const std::string& eventJson)
{
    Json::Value errorValue;

    if (!eventJson.empty()) {
        auto maybeEventJsonParsed = tryParseJson(eventJson);
        if (maybeEventJsonParsed) {
            errorValue = std::move(*maybeEventJsonParsed);
        } else {
            // Cannot use error logging: may induce new LogError events
            errorValue["bad_event"] = eventJson;
        }
    }

    errorValue["log"] = message;
    errorValue["file_name"] = sourceFileName;
    errorValue["file_line"] = sourceLine;

    reportError("log_error", quasar::jsonToString(errorValue));
}

void MetricaConnector::reportKeyValues(const std::string& eventName,
                                       const std::unordered_map<std::string, std::string>& keyValues, ITelemetry::Flags flags)
{
    if (isOverflowed(eventName)) {
        return;
    }

    QuasarMessage message;
    message.mutable_metrica_message()->set_flags(flags);
    auto reportKeyValue = message.mutable_metrica_message()->mutable_report_key_value();
    reportKeyValue->set_event_name(TString(eventName));
    for (const auto& keyValue : keyValues)
    {
        (*reportKeyValue->mutable_key_values())[keyValue.first] = keyValue.second;
    }

    enqueue(message);
}

void MetricaConnector::enqueue(const quasar::proto::QuasarMessage& message)
{
    if (!message.has_metrica_message()) {
        YIO_LOG_ERROR_EVENT("MetricaConnector.MalformedMessage", "Malformed metrica message");
        return;
    }

    if (isEventMessage(message.metrica_message())) {
        if (!messagesToSend_.tryPush(message)) {
            dailyStats_.increment({Counters::TOTAL, Counters::QUEUE_DROP});
        } else {
            dailyStats_.increment(Counters::TOTAL);
        }
    } else {
        messagesToSend_.tryPush(message);
    }
}

std::shared_ptr<const YandexIO::LatencyData> MetricaConnector::createLatencyPoint() {
    return std::make_shared<const YandexIO::LatencyData>();
}

void MetricaConnector::reportLatency(std::shared_ptr<const YandexIO::LatencyData> latencyData, const std::string& eventName) {
    if (latencyData) {
        const auto diff = latencyData->msPassedTillNow();
        Json::Value json = Json::objectValue;
        json["value"] = diff;
        reportEvent(eventName, jsonToString(json));
    } else {
        YIO_LOG_WARN("Trying to report null latency of " << eventName);
    }
}

void MetricaConnector::reportLatency(std::shared_ptr<const YandexIO::LatencyData> latencyData, const std::string& eventName, const std::string& eventJson) {
    if (latencyData) {
        const auto diff = latencyData->msPassedTillNow();
        Json::Value json = tryParseJsonOrEmpty(eventJson);
        json["value"] = diff;
        reportEvent(eventName, jsonToString(json));
    } else {
        YIO_LOG_WARN("Trying to report null latency of " << eventName);
    }
}

void MetricaConnector::putAppEnvironmentValue(const std::string& key, const std::string& value)
{
    {
        std::scoped_lock lock(environmentVariablesMutex_);
        environmentVariables_[key] = value;
    }
    QuasarMessage message;
    message.mutable_metrica_message()->mutable_app_environment_value()->set_key(TString(key));
    message.mutable_metrica_message()->mutable_app_environment_value()->set_value(TString(value));
    enqueue(message);
}

void MetricaConnector::deleteAppEnvironmentValue(const std::string& key) {
    {
        std::scoped_lock lock(environmentVariablesMutex_);
        environmentVariables_.erase(key);
    }
    QuasarMessage message;
    message.mutable_metrica_message()->set_delete_environment_key(TString(key));
    enqueue(message);
}

void MetricaConnector::setRateLimiter(const std::string& jsonConfig) {
    std::unique_ptr<quasar::BucketRateLimiter> newRateLimiter;
    auto configOpt = tryParseJson(jsonConfig);

    Json::Value config;
    if (configOpt) {
        config = *configOpt;
    }

    YIO_LOG_INFO("Creating new rate limiter with params: " << config);
    newRateLimiter = createRateLimiter(config);

    {
        std::scoped_lock guard(rateLimiterMutex_);

        rateLimiter_ = std::move(newRateLimiter);
        if (rateLimiter_) {
            rateLimiter_->start();
        }
        YIO_LOG_INFO("New rate limiter started");
    }
}

bool MetricaConnector::isOverflowed(const std::string& event) {
    std::scoped_lock guard(rateLimiterMutex_);

    if (rateLimiter_ &&
        rateLimiter_->addEvent(event) == BucketRateLimiter::OverflowStatus::OVERFLOWED) {
        dailyStats_.increment(Counters::RATELIMIT_DROP);
        return true;
    }

    return false;
}

void MetricaConnector::requestUUID(UUIDCallback cb) {
    std::scoped_lock lock(uuidMutex_);

    if (uuid_) {
        cb(*uuid_);
    } else {
        uuidCallbacks_.push_back(cb);
    }
}

void MetricaConnector::setSenderName(const std::string& name) {
    senderName_ = name;
}

MetricaConnector::Params::Params(MetricaConnector* connector)
    : connector_(connector)
{
}

void MetricaConnector::Params::setNetworkStatus(NetworkStatus status) {
    QuasarMessage message;
    message.mutable_metrica_message()->mutable_network_status()->CopyFrom(toProtoNetworkStatus(status));
    connector_->enqueue(message);
}

void MetricaConnector::Params::setWifiNetworks(const std::vector<WifiNetwork>& networks) {
    QuasarMessage message;
    auto wifi_list = message.mutable_metrica_message()->mutable_wifi_list(); /* https://st.yandex-team.ru/ALICE-10046 */
    for (const auto& network : networks) {
        auto hotspot = wifi_list->add_hotspots();
        hotspot->set_mac(TString(network.mac));
        hotspot->set_rssi(network.rssi);
        hotspot->set_ssid(TString(network.ssid));
        hotspot->set_is_connected(network.isConnected);
    }
    connector_->enqueue(message);
}

void MetricaConnector::Params::setLocation(double lat, double lon) {
    QuasarMessage message;
    message.mutable_metrica_message()->mutable_location()->set_latitude(lat);
    message.mutable_metrica_message()->mutable_location()->set_longitude(lon);
    connector_->enqueue(message);
}

void MetricaConnector::Params::setTimezone(const std::string& name, int32_t offsetSec) {
    QuasarMessage message;
    message.mutable_metrica_message()->mutable_timezone()->set_timezone_name(TString(name));
    message.mutable_metrica_message()->mutable_timezone()->set_timezone_offset_sec(offsetSec);
    connector_->enqueue(message);
}

void MetricaConnector::Params::setConfig(const std::string& config) {
    QuasarMessage message;
    message.mutable_metrica_message()->set_config(TString(config));
    connector_->enqueue(message);
}

YandexIO::ITelemetry::IParams* MetricaConnector::params() {
    return params_.get();
}

MetricaConnector::~MetricaConnector() {
    stopped_ = true;
    messagesToSend_.push(StopQueue());
    senderCV_.notify_all();
    senderThread_.join();

    connector_->shutdown();
}

void MetricaConnector::senderThread() {
    QueueElement queueElement;
    while (!stopped_) {
        messagesToSend_.pop(queueElement);
        QuasarMessage message;
        try {
            message = std::get<QuasarMessage>(queueElement);
        } catch (const std::bad_variant_access& e) {
            // We got StopEvent
            break;
        }

        while (!stopped_ && !sendWithAcknowledge(std::move(message))) {
            std::unique_lock<std::mutex> lock(senderMutex_);
            senderCV_.wait_for(lock, std::chrono::milliseconds(retryTimeoutMs_.load()), [this]() { return stopped_.load(); });
            message = std::get<QuasarMessage>(queueElement); // refresh message object
        }
    }
}

bool MetricaConnector::sendWithAcknowledge(quasar::proto::QuasarMessage&& message) {
    try {
        connector_->sendRequestSync(std::move(message), std::chrono::seconds(1));
        return true;
    } catch (const std::logic_error& e) {
        YIO_LOG_ERROR_EVENT("MetricaConnector.SendWithAcknowledge", "Exception: " << e.what());
        return false;
    } catch (const std::exception& e) {
        YIO_LOG_WARN("Exception in sendWithAcknowledge: " << e.what());
        return false;
    }
}

bool MetricaConnector::waitUntilConnected(std::chrono::seconds time) {
    return connector_->waitUntilConnected(time);
}

void MetricaConnector::waitUntilConnected() {
    connector_->waitUntilConnected();
}

bool MetricaConnector::isOnConnectedCalled() const {
    return isOnConnectedCalled_.load();
}

void MetricaConnector::setRetryTimeoutMs(int retryTimeoutMs) {
    retryTimeoutMs_.store(retryTimeoutMs);
}

void MetricaConnector::waitUntilDisconnected() const {
    connector_->waitUntilDisconnected();
}
