#pragma once

#include <yandex_io/interfaces/user_config/i_user_config_provider.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/rate_limiter/bucket/bucket_rate_limiter.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/threading/blocking_queue.h>
#include <yandex_io/libs/threading/lifetime.h>
#include <yandex_io/libs/threading/periodic_executor.h>
#include <yandex_io/libs/counters/default_daily_counter.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/protos/quasar_proto_forward.h>

#include <atomic>
#include <map>
#include <string>
#include <unordered_map>
#include <variant>

namespace quasar {

    class MetricaConnector: public YandexIO::ITelemetry {
    public:
        MetricaConnector(const std::shared_ptr<ipc::IIpcFactory>& ipcFactory, const std::string& serviceName);
        ~MetricaConnector();

        void reportEvent(const std::string& event, ITelemetry::Flags flags = 0) override;
        void reportEvent(const std::string& event, const std::string& eventJson, ITelemetry::Flags flags = 0) override;
        void reportError(const std::string& errorEventName, ITelemetry::Flags flags = 0) override;
        // will produce java-style error
        void reportError(const std::string& errorEventName, const std::string& errorValue, ITelemetry::Flags flags = 0) override;
        void reportLogError(const std::string& message, const std::string& sourceFileName, size_t sourceLine, const std::string& eventJson) override;
        void reportKeyValues(const std::string& eventName, const std::unordered_map<std::string, std::string>& keyValues, ITelemetry::Flags flags = 0) override;

        std::shared_ptr<const YandexIO::LatencyData> createLatencyPoint() override;
        void reportLatency(std::shared_ptr<const YandexIO::LatencyData> latencyData, const std::string& eventName) override;
        void reportLatency(std::shared_ptr<const YandexIO::LatencyData> latencyData, const std::string& eventName, const std::string& eventJson) override;

        void putAppEnvironmentValue(const std::string& key, const std::string& value) override;
        void deleteAppEnvironmentValue(const std::string& key) override;

        void setRateLimiter(const std::string& jsonConfig) override;

        void requestUUID(UUIDCallback cb) override;
        void setSenderName(const std::string& name) override;

        class Params: public YandexIO::ITelemetry::IParams {
        public:
            Params(MetricaConnector* connector);

            void setNetworkStatus(NetworkStatus status) override;
            void setWifiNetworks(const std::vector<WifiNetwork>& networks) override;
            void setLocation(double lat, double lon) override;
            void setTimezone(const std::string& name, int32_t offsetSec) override;
            void setConfig(const std::string& config) override;

        private:
            MetricaConnector* connector_;
        };

        IParams* params() override;

        bool waitUntilConnected(std::chrono::seconds time);
        void waitUntilConnected();
        void waitUntilDisconnected() const;
        // a way for tests to known that "onConnected" ipc callback was fired
        bool isOnConnectedCalled() const;

        void setRetryTimeoutMs(int retryTimeoutMs);

    private:
        bool isOverflowed(const std::string& eventName);

        void handleReconnect();

        void handleQuasarMessage(const ipc::SharedMessage& message);

    private:
        class StopQueue {};
        using QueueElement = std::variant<quasar::proto::QuasarMessage, StopQueue>;

        void enqueue(const quasar::proto::QuasarMessage& message);
        void senderThread();
        bool sendWithAcknowledge(quasar::proto::QuasarMessage&& message);
        std::atomic_bool isOnConnectedCalled_{false};
        std::atomic_bool stopped_{false};
        std::string senderName_;

        BlockingQueue<QueueElement> messagesToSend_;

        std::thread senderThread_;

        std::mutex environmentVariablesMutex_;
        std::map<std::string, std::string> environmentVariables_;

        std::mutex senderMutex_;
        SteadyConditionVariable senderCV_;

        std::shared_ptr<ipc::IConnector> connector_;

        std::atomic_int retryTimeoutMs_{10000};

        std::mutex uuidMutex_;
        std::optional<std::string> uuid_;
        std::vector<UUIDCallback> uuidCallbacks_;

        std::unique_ptr<Params> params_;

        std::mutex rateLimiterMutex_;
        std::unique_ptr<quasar::BucketRateLimiter> rateLimiter_;

        enum Counters {
            TOTAL,
            QUEUE_DROP,
            RATELIMIT_DROP,
            MAX_
        };

        DefaultDailyCounter<Counters> dailyStats_;
        std::unique_ptr<PeriodicExecutor> periodicExecutor_;
    };

} // namespace quasar
