#include "metrics_collector.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/metrics_collector/consumers/json/json_metric_consumer.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/protos/enum_names/enum_names.h>

#include <boost/algorithm/string.hpp>

#include <linux/i2c-dev.h>
#include <linux/i2c.h>

#include <cctype>
#include <fstream>
#include <ios>
#include <iterator>
#include <sstream>
#include <string>
#include <unordered_set>
#include <vector>

#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/statfs.h>
#include <sys/types.h>

YIO_DEFINE_LOG_MODULE("metrics_collector");

#define THROTTLED_LOG_ERROR(collector, tag, message)  \
    {                                                 \
        std::ostringstream __os;                      \
        __os << message;                              \
        if ((collector).shouldLogError(__os.str())) { \
            YIO_LOG_ERROR_EVENT(tag, message);        \
        }                                             \
    }

using namespace quasar;
using namespace quasar::proto;

const std::string MetricsCollectorBase::NetworkStatistics::IFACES_STATS_FILENAME = "/proc/net/dev";
const std::string MetricsCollectorBase::NetworkStatistics::ROUTING_TABLE_FILENAME = "/proc/net/route";
const std::string MetricsCollectorBase::NetworkStatistics::INTERFACES_DIRECTORY = "/sys/class/net";
const std::string MetricsCollectorBase::NetworkStatistics::OPERSTATE_FILENAME = "operstate";
const std::string MetricsCollectorBase::NetworkStatistics::DEFAULT_GATEWAY = "00000000";
const int MetricsCollectorBase::NetworkStatistics::ROUTING_TABLE_MIN_FIELDS_COUNT = 3;
const int MetricsCollectorBase::NetworkStatistics::IFACES_STATS_HEADERS_COUNT = 3;

namespace {
    const std::size_t MAX_LAST_ERROR_TIMES = 1024;
    const std::chrono::minutes LOG_ERROR_INTERVAL = std::chrono::minutes(60);
} // namespace

MetricsCollectorBase::MetricsCollectorBase(std::shared_ptr<YandexIO::IDevice> device, const std::shared_ptr<ipc::IIpcFactory>& ipcFactory,
                                           std::shared_ptr<PingManager> pingManager, ThermalNodes thermalNodes)
    : device_(std::move(device))
    , pingManager_(std::move(pingManager))
    , thermalNodes_(std::move(thermalNodes))
    , disksList_{[this] {
        std::vector<std::string> res;
        Json::Value data = tryGetArray(device_->configuration()->getServiceConfig("monitord"), "disksList", Json::Value(Json::arrayValue));
        for (const auto& d : data)
        {
            try {
                auto disk = d.asString();
                if (!disk.empty())
                {
                    res.emplace_back(std::move(disk));
                }
            } catch (const std::exception&) {
                YIO_LOG_ERROR_EVENT("MetricsCollector.InvalidDiskList", "Invalid value in quasar.cfg for field monitord/disksList");
            }
        }
        return res;
    }()}
    , deviceContext_(ipcFactory)
{
    coreCount_ = sysconf(_SC_NPROCESSORS_ONLN);
    pageSizeKb_ = sysconf(_SC_PAGESIZE) / 1024;

    auto config = device_->configuration()->getServiceConfig("monitord");

    auto servicesListString = config["servicesList"].asString();
    if (!servicesListString.empty()) {
        boost::split(services_, servicesListString, boost::is_any_of(","));
    }

    pingManager_->start(config["pinger"]);
    configurePropertiesReport(config["collectSystemProperties"]);

    for (int status = NetworkStatus_Status_Status_MIN; status <= NetworkStatus_Status_Status_MAX; ++status) {
        networkStatusesUsageMs_[networkStatusStatusName((NetworkStatus_Status)status)] = 0;
    }
    lastNetworkCheck_ = std::chrono::steady_clock::now();

    deviceContext_.onNumericMetrics = [this](const std::string& key, double value) {
        std::lock_guard lock(mutex_);
        platformAggregateStatMetrics_.addDGauge(key, value);
    };
    // TODO[labudidabudai] не уверен, что это лучший способ обрабатывать категориальные данные, возможно надо по-другому
    deviceContext_.onCategoricalMetrics = [this](const std::string& key, const std::string& value) {
        std::lock_guard lock(mutex_);
        platformCategoricalMetrics_[key].push_back(value);
    };
}

void MetricsCollectorBase::setConfig(const Json::Value& config) {
    const auto& systemConfig = config["system_config"];

    if (const auto& pingerConfig = systemConfig["pinger"]; !pingerConfig.isNull()) {
        pingManager_->start(pingerConfig);
    } else {
        const auto config = device_->configuration()->getServiceConfig("monitord");
        pingManager_->start(config["pinger"]);
    }
}

void MetricsCollectorBase::setNetworkStatus(const proto::NetworkStatus& status) {
    std::lock_guard guard(mutex_);
    const auto now = std::chrono::steady_clock::now();
    networkStatusesUsageMs_[networkStatusStatusName(currentNetworkStatus_)] += std::chrono::duration_cast<std::chrono::milliseconds>(now - lastNetworkCheck_).count();
    lastNetworkCheck_ = now;
    if (currentNetworkStatus_ != status.status() || currentConnectionType_ != status.type()) {
        pingManager_->reloadGateway();

        currentNetworkStatus_ = status.status();
        currentConnectionType_ = status.type();
    }
}

MetricsCollectorBase::~MetricsCollectorBase()
{
    executor_.reset(nullptr);
}

void MetricsCollectorBase::configurePropertiesReport(const Json::Value& propertiesReportConfig) {
    if (!propertiesReportConfig.isArray()) {
        return;
    }
    auto size = propertiesReportConfig.size();
    collectedProperties_.clear();
    for (Json::ArrayIndex i = 0; i < size; i++) {
        collectedProperties_.emplace_back(propertiesReportConfig[i].asString());
    }
}

Json::Value MetricsCollectorBase::getNetworkMetrics() {
    std::lock_guard<std::mutex> lock(mutex_);
    return getNetworkMetricsLocked();
}

Json::Value MetricsCollectorBase::getNetworkMetricsLocked() {
    std::unordered_map<std::string, NumericStat<double>> pingDurations;

    Json::Value pings;

    auto windowPingStats = pingManager_->getRecentPingsStatistics();

    for (auto it = windowPingStats.begin(); it != windowPingStats.end(); ++it) {
        const auto& st = it->second;
        const std::string host = st.host;

        pings[host]["gateway"] = st.isGateway;

        for (const auto& ping : st.pingDurations) {
            pingDurations[host].process(ping);
        }

        pings[host]["sent"] = tryGetUInt64(pings[host], "sent", 0) + st.packetsSent;
        pings[host]["recieved"] = tryGetUInt64(pings[host], "recieved", 0) + st.packetsReceived;
        pings[host]["lost"] = tryGetUInt64(pings[host], "lost", 0) + st.packetsLost;
        pings[host]["duplicate"] = tryGetUInt64(pings[host], "duplicate", 0) + st.packetsDuplicate;
    }

    for (const auto& ping : pingDurations) {
        pings[ping.first]["min"] = ping.second.getMin();
        pings[ping.first]["max"] = ping.second.getMax();
        pings[ping.first]["mean"] = ping.second.getMean();
        pings[ping.first]["last"] = ping.second.getLast();
    }

    Json::Value metrics;

    metrics["pings"] = pings;

    auto networkStatistics = NetworkStatistics();
    networkStatistics.collect(*this);
    metrics["network_diagnostic"] = networkStatistics.toJson();

    return metrics;
}

Json::Value MetricsCollectorBase::getMetrics()
{
    std::lock_guard<std::mutex> lock(mutex_);

    Json::Value metricsJson = getNetworkMetricsLocked();

    {
        JsonMetricConsumer jsonPlatformNumericMetricConsumer(metricsJson["platformValues"]["numeric"]);
        platformAggregateStatMetrics_.flush(jsonPlatformNumericMetricConsumer);

        auto& platformCategoricalValues = metricsJson["platformValues"]["categorical"];
        for (const auto& metric : platformCategoricalMetrics_) {
            platformCategoricalValues[metric.first] = Json::arrayValue;
            int counter = 0;
            for (const auto& value : metric.second) {
                platformCategoricalValues[metric.first][counter] = value;
                ++counter;
            }
        }

        platformCategoricalMetrics_.clear();
    }

    {
        // Send network statuses statistics for this period and set all to zero for the next period
        const auto now = std::chrono::steady_clock::now();
        networkStatusesUsageMs_[networkStatusStatusName(currentNetworkStatus_)] += std::chrono::duration_cast<std::chrono::milliseconds>(now - lastNetworkCheck_).count();
        lastNetworkCheck_ = now;
        for (auto& key : networkStatusesUsageMs_) {
            metricsJson["wifiStatusStatistics"][key.first] = key.second;
            key.second = 0;
        }
    }

    auto partitionsInfo = getPartitionsInfo();

    Json::Value partitions = Json::arrayValue;
    for (const auto& partition : partitionsInfo)
    {
        Json::Value object;
        object["path"] = partition.path;
        object["freeBytes"] = partition.freeBytes;
        object["totalBytes"] = partition.totalBytes;
        partitions.append(std::move(object));
    }

    metricsJson["partitions"] = partitions;

    {
        JsonMetricConsumer jsonNumericMetricConsumer(metricsJson);
        aggregateStatMetrics_.flush(jsonNumericMetricConsumer);
    }

    Json::Value systemProperties;
    for (const std::pair<std::string, std::string> element : collectedPropertiesValues_) {
        systemProperties[element.first] = element.second;
    }
    metricsJson["systemProperties"] = systemProperties;

    return metricsJson;
}

void MetricsCollectorBase::collectCPUStats(IMetricConsumer& metrics, double timeElapsedMs) {
    try {
        const CPUStats cpuStats = getCPUStats();

        if (lastCPUStats_.idleMs != 0 && timeElapsedMs >= 1)
        {
            metrics.addDGauge("cpuIdlePercent", 100.0 * (cpuStats.idleMs - lastCPUStats_.idleMs) / timeElapsedMs);
            metrics.addDGauge("cpuUserPercent", 100.0 * (cpuStats.userMs - lastCPUStats_.userMs) / timeElapsedMs);
            metrics.addDGauge("cpuSystemPercent", 100.0 * (cpuStats.systemMs - lastCPUStats_.systemMs) / timeElapsedMs);
            metrics.addDGauge("cpuNicePercent", 100.0 * (cpuStats.niceMs - lastCPUStats_.niceMs) / timeElapsedMs);
            metrics.addDGauge("cpuIoWaitPercent", 100.0 * (cpuStats.ioWaitMs - lastCPUStats_.ioWaitMs) / timeElapsedMs);
        }

        lastCPUStats_ = cpuStats;
    } catch (const std::runtime_error& e) {
        THROTTLED_LOG_ERROR(*this, "MetricsCollector.CollectCpuStats.Exception", "Cannot collect CPU stats: " << e.what());
    }
}

void MetricsCollectorBase::collectServicesInfo(IMetricConsumer& metrics, double timeElapsedMs) {
    const QuasarServicesInfo servicesInfo = getServicesInfo();

    if (!lastServicesInfo_.empty() && timeElapsedMs >= 1)
    {
        for (const auto& serviceName : services_) {
            const auto& info = servicesInfo.get(serviceName.c_str());
            const auto& lastInfo = lastServicesInfo_.get(serviceName.c_str());
            metrics.addIGauge(serviceName + std::string("_vszKbytes"), info.vszKbytes);
            metrics.addIGauge(serviceName + std::string("_rssKbytes"), info.rssKbytes);
            metrics.addIGauge(serviceName + std::string("_majorPageFaults"), info.majorPageFaults);
            if (info.runningThreads > 0) {
                metrics.addIGauge(serviceName + std::string("_runningThreads"), info.runningThreads);
            }
            if (info.openFdCount > 0) {
                metrics.addIGauge(serviceName + std::string("_openFdCount"), info.openFdCount);
            }

            double cpuUserPercent = 100.0 * (info.utimeMs - lastInfo.utimeMs) / timeElapsedMs;
            double cpuSystemPercent = 100.0 * (info.stimeMs - lastInfo.stimeMs) / timeElapsedMs;
            if (cpuUserPercent < 0 || cpuSystemPercent < 0) {
                YIO_LOG_INFO("Service " << serviceName << " has incorrect stats (most probably due to restart of service), skipping");
                Json::Value failedService;
                failedService["failedService"] = serviceName;
                device_->telemetry()->reportEvent("collectServiceCpuMetricsFail", jsonToString(failedService));
                continue;
            }

            metrics.addDGauge(serviceName + std::string("_cpuUserPercent"), cpuUserPercent);
            metrics.addDGauge(serviceName + std::string("_cpuSystemPercent"), cpuSystemPercent);
        }
    }

    lastServicesInfo_ = servicesInfo;
}

void MetricsCollectorBase::collectThermalStats(IMetricConsumer& metrics) {
    const auto thermalStats = getThermalNodesStats();

    for (const auto& thermalStat : thermalStats) {
        metrics.addDGauge(std::string("temperature_") + thermalStat.thermalNodeName, thermalStat.temperature);
    }
}

void MetricsCollectorBase::collectMemStats(IMetricConsumer& metrics) {
    try {
        const MemInfo memInfo = getMemInfo();

        metrics.addIGauge("totalRAMBytes", memInfo.totalBytes);
        metrics.addIGauge("freeRAMBytes", memInfo.freeBytes);
        metrics.addIGauge("totalSwapBytes", memInfo.totalSwapBytes);
        metrics.addIGauge("freeSwapBytes", memInfo.freeSwapBytes);
        metrics.addIGauge("cachedBytes", memInfo.cachedBytes);
        metrics.addIGauge("vmallocUsedBytes", memInfo.vmallocUsedBytes);
        metrics.addIGauge("slabBytes", memInfo.slabBytes);
        metrics.addIGauge("mappedBytes", memInfo.mappedBytes);
        metrics.addIGauge("majorPageFaults", memInfo.majorPageFaults);
    } catch (const std::runtime_error& e) {
        THROTTLED_LOG_ERROR(*this, "MetricsCollector.CollectMemStats.Exception", "Cannot collect mem stats: " << e.what());
    }
}

void MetricsCollectorBase::collectUptime(IMetricConsumer& metrics) {
    metrics.addIGauge("uptimeSeconds", uptime());
}

void MetricsCollectorBase::collectLoadAverage(IMetricConsumer& metrics) {
    try {
        metrics.addDGauge("loadAverage1min", strtod(getFileContent("/proc/loadavg").c_str(), nullptr));
    } catch (const std::runtime_error& e) {
        THROTTLED_LOG_ERROR(*this, "MetricsCollector.CollectLA.Exception", "Cannot collect load average: " << e.what());
    }
}

void MetricsCollectorBase::collectBaseMetrics(IMetricConsumer& metrics)
{
    double timeElapsedMs = -1;
    const auto now = SteadyClock::now();
    if (lastMonotonicTimePoint_.has_value()) {
        const double timeDiff = std::chrono::duration_cast<std::chrono::milliseconds>(now - lastMonotonicTimePoint_.value()).count();
        timeElapsedMs = timeDiff * coreCount_;
    }
    lastMonotonicTimePoint_ = now;

    collectCPUStats(metrics, timeElapsedMs);
    collectMemStats(metrics);
    collectServicesInfo(metrics, timeElapsedMs);
    collectThermalStats(metrics);
    collectUptime(metrics);
    collectLoadAverage(metrics);
}

void MetricsCollectorBase::collectPlatformMetrics(IMetricConsumer& /* metrics */) {
    // Must be overriden in platform implementations
}

void MetricsCollectorBase::collectSystemMetrics() {
    std::lock_guard lock(mutex_);

    try {
        collectBaseMetrics(aggregateStatMetrics_);
        collectPlatformMetrics(aggregateStatMetrics_);
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("MetricsCollector.CollectSystemMetrics.Exception", "Error in collectSystemMetrics: " << e.what());
    }
}

MetricsCollectorBase::CPUStats MetricsCollectorBase::getCPUStats() const {
    std::ifstream stat("/proc/stat");
    if (!stat.good())
    {
        throw std::runtime_error("Cannot open /proc/stat");
    }
    std::string line;
    std::getline(stat, line);

    std::vector<std::string> fields;

    boost::iter_split(fields, line, boost::first_finder(" "));
    if (fields.size() < 6)
    {
        throw std::runtime_error("Cannot get idle from /proc/stat. Not enough fields (" + line + ")");
    }

    CPUStats result;
    result.userMs = atoi(fields[2].c_str()) * 10;
    result.niceMs = atoi(fields[3].c_str()) * 10;
    result.systemMs = atoi(fields[4].c_str()) * 10;
    result.idleMs = atoi(fields[5].c_str()) * 10;
    result.ioWaitMs = atoi(fields[6].c_str()) * 10;

    return result;
}

MetricsCollectorBase::MemInfo MetricsCollectorBase::getMemInfo() const {
    std::string token;
    MemInfo result{};
    int64_t* ptr = nullptr;

    std::ifstream meminfo("/proc/meminfo");
    if (!meminfo.good()) {
        throw std::runtime_error("Cannot open /proc/meminfo");
    }

    while (meminfo >> token)
    {
        ptr = nullptr;
        if ("MemTotal:" == token)
            ptr = &result.totalBytes;
        if ("MemFree:" == token)
            ptr = &result.freeBytes;
        if ("Cached:" == token)
            ptr = &result.cachedBytes;
        if ("SwapTotal:" == token)
            ptr = &result.totalSwapBytes;
        if ("SwapFree:" == token)
            ptr = &result.freeSwapBytes;
        if ("VmallocUsed:" == token)
            ptr = &result.vmallocUsedBytes;
        if ("Slab:" == token)
            ptr = &result.slabBytes;
        if ("Mapped:" == token)
            ptr = &result.mappedBytes;
        if (ptr != nullptr)
        {
            meminfo >> *ptr;
            *ptr *= 1024L;
        }
    }

    std::ifstream vmstat("/proc/vmstat");
    if (!vmstat.good()) {
        throw std::runtime_error("Cannot open /proc/vmstat");
    }

    while (vmstat >> token)
    {
        ptr = nullptr;
        if ("pgmajfault" == token)
            ptr = &result.majorPageFaults;
        if (ptr != nullptr)
            vmstat >> *ptr;
    }

    return result;
}

MetricsCollectorBase::QuasarServicesInfo::ServiceInfo MetricsCollectorBase::QuasarServicesInfo::collectServiceInfo(const MetricsCollectorBase& collector, const std::string& proc_dir_path, int pageSizeKb) const {
    MetricsCollectorBase::QuasarServicesInfo::ServiceInfo info{};
    {
        std::ifstream proc_stat;
        proc_stat.open(proc_dir_path + "/stat");
        if (proc_stat.good()) {
            std::vector<std::string> stat_fields;
            std::string line;

            std::getline(proc_stat, line);
            boost::iter_split(stat_fields, line, boost::first_finder(" "));
            if (stat_fields.size() < MetricsCollectorBase::QuasarServicesInfo::LAST_EXPECTED) {
                YIO_LOG_ERROR_EVENT("MetricsCollector.CollectServicesStats.NotEnoughStatFields", "Not enogh data in \"stat\". stat_fields size: " << stat_fields.size() << " Data: " << line);
            } else {
                info.majorPageFaults =
                    atoll(stat_fields[MetricsCollectorBase::QuasarServicesInfo::MAJOR_FAULTS_IDX].c_str());
                info.utimeMs =
                    atoll(stat_fields[MetricsCollectorBase::QuasarServicesInfo::UTIME_IDX].c_str()) * 10;
                info.stimeMs =
                    atoll(stat_fields[MetricsCollectorBase::QuasarServicesInfo::STIME_IDX].c_str()) * 10;
                info.runningThreads =
                    atoll(stat_fields[MetricsCollectorBase::QuasarServicesInfo::NUM_THREADS_IDX].c_str());
                info.vszKbytes =
                    atoll(stat_fields[MetricsCollectorBase::QuasarServicesInfo::VMSIZE_BYTES].c_str()) / 1024;
                info.rssKbytes =
                    atoll(stat_fields[MetricsCollectorBase::QuasarServicesInfo::RSS_PAGES].c_str()) * pageSizeKb;
            }
        }
    }

    try {
        const auto fds = getDirectoryFileList(proc_dir_path + "/fd"); // '..' and '.' arent returned here
        if (!fds.empty()) {
            info.openFdCount = fds.size();
        }
    } catch (const ErrnoException& e) {
        THROTTLED_LOG_ERROR(collector, "MetricsCollector.CollectServicesFdCount.ErrnoException", "Can't count Open Files. Error: " << e.what());
    }

    return info;
}

MetricsCollectorBase::QuasarServicesInfo MetricsCollectorBase::getServicesInfo() const {
    MetricsCollectorBase::QuasarServicesInfo servicesInfo;
    std::vector<std::string> proc_content = getDirectoryFileList("/proc");

    for (const auto& proc_entry : proc_content)
    {
        const std::string proc_dir_path("/proc/" + proc_entry);

        const bool path_valid = std::all_of(proc_entry.begin(), proc_entry.end(), [](char c) { return (::isdigit(c)); });

        struct stat st;
        if (lstat(proc_dir_path.c_str(), &st) < 0 || !S_ISDIR(st.st_mode) || !path_valid) {
            continue;
        }

        std::ifstream proc_cmdline_file(proc_dir_path + "/cmdline");
        if (!proc_cmdline_file.good())
            continue;

        std::string proc_cmdline;
        proc_cmdline_file >> proc_cmdline;

        const char* service_name = matchProcWithService(proc_cmdline, services_);

        if (service_name == nullptr) {
            continue;
        }

        MetricsCollectorBase::QuasarServicesInfo::ServiceInfo info;

        info = servicesInfo.collectServiceInfo(*this, proc_dir_path, pageSizeKb_);

        /* NOTE: Need to store a POINTER to a std::string from services array. This array is constant and static, so user
         * side will try to get services info by POINTERS
         */
        servicesInfo.store(service_name, info);
    }

    return servicesInfo;
}

std::vector<MetricsCollectorBase::PartitionInfo> MetricsCollectorBase::getPartitionsInfo() const {
    std::ifstream mounts("/proc/mounts");
    std::string line;
    std::vector<PartitionInfo> result;
    while (std::getline(mounts, line))
    {
        std::vector<std::string> fields;
        boost::iter_split(fields, line, boost::first_finder(" "));
        if (fields.size() < 2)
            continue;

        std::string name = fields[0];

        struct statfs st;
        if (statfs(fields[1].c_str(), &st) < 0)
        {
            THROTTLED_LOG_ERROR(*this, "MetricsCollector.CollectPartitionsInfo.StatfsFailed", "Cannot do statfs for " << fields[1] << ": " << strError(errno) << " (" << errno << ")");
            continue;
        }
        if (st.f_blocks == 0)
            continue;

        PartitionInfo info;
        info.path = fields[1];
        info.totalBytes = ((int64_t)st.f_blocks * st.f_bsize);
        info.freeBytes = (int64_t)st.f_bfree * st.f_bsize;

        result.push_back(std::move(info));
    }

    return result;
}

std::vector<MetricsCollectorBase::ThermalStat> MetricsCollectorBase::getThermalNodesStats() {
    std::vector<ThermalStat> res;
    for (const auto& thermalNode : thermalNodes_) {
        try {
            ThermalStat thermalStat;
            thermalStat.thermalNodeName = thermalNode.second.first;
            thermalStat.temperature = strtod(getFileContent(thermalNode.first.c_str()).c_str(), nullptr) * thermalNode.second.second;
            res.push_back(thermalStat);
        } catch (const std::runtime_error& e) {
            THROTTLED_LOG_ERROR(*this, "MetricsCollector.CollectThermalStatus.Exception", "Cannot collect thermal node '" << thermalNode.second.first << "' stats: " << e.what());
        }
    }
    return res;
}

MetricsCollectorBase::NetworkStatistics::NetworkStatistics() {
    interfacesWhiteList_ = {"lo", "wlan", "eth"};
    dataWhiteList_ = {"bytes", "errs"};
}

void MetricsCollectorBase::NetworkStatistics::collect(const MetricsCollectorBase& collector) {
    if (fileExists(ROUTING_TABLE_FILENAME)) {
        std::ifstream routingTable(ROUTING_TABLE_FILENAME);
        defaultIfaceName_ = getDefaultInterfaceName(routingTable);
    } else {
        THROTTLED_LOG_ERROR(collector, "MetricsCollector.CollectNetworkStatistics.RountingTableDoNotExist", "Cannot collect NetworkStatistics. File '" << ROUTING_TABLE_FILENAME << "' doesn't exists")
    }

    if (fileExists(IFACES_STATS_FILENAME)) {
        std::ifstream ifacesStats(IFACES_STATS_FILENAME);
        interfaces_ = getInterfacesInfo(ifacesStats);
    } else {
        THROTTLED_LOG_ERROR(collector, "MetricsCollector.CollectNetworkStatistics.IfaceDoNotExist", "Cannot collect NetworkStatistics. File '" << IFACES_STATS_FILENAME << "' doesn't exists")
    }

    for (auto& iface : interfaces_) {
        const std::string filename = getOperstateFilename(iface.name);
        if (fileExists(filename)) {
            std::ifstream operstate(filename);
            iface.operstate = getOperstate(operstate);
        } else {
            THROTTLED_LOG_ERROR(collector, "MetricsCollector.CollectNetworkStatistics.IfaceOperstateDoNotExist", "Cannot collect NetworkStatistics. File '" << filename << "' doesn't exists")
        }
    }
}

Json::Value MetricsCollectorBase::NetworkStatistics::toJson() const {
    Json::Value networkDiagnostic = Json::objectValue;
    networkDiagnostic["defaultInterface"] = defaultIfaceName_;

    Json::Value interfacesJson = Json::arrayValue;
    for (const InterfaceInfo& interface : interfaces_) {
        if (!isInterfaceInWhiteList(interface.name)) {
            continue;
        }
        Json::Value interfaceJson;
        interfaceJson["name"] = interface.name;
        interfaceJson["operstate"] = interface.operstate;

        for (const auto& dataName : dataWhiteList_) {
            if (interface.transmitStatistics.find(dataName) != interface.transmitStatistics.end()) {
                interfaceJson["tx_" + dataName] = interface.transmitStatistics.at(dataName);
            } else {
                interfaceJson["tx_" + dataName] = 0;
            }

            if (interface.receiveStatistics.find(dataName) != interface.receiveStatistics.end()) {
                interfaceJson["rx_" + dataName] = interface.receiveStatistics.at(dataName);
            } else {
                interfaceJson["rx_" + dataName] = 0;
            }
        }
        interfacesJson.append(std::move(interfaceJson));
    }
    networkDiagnostic["interfaces"] = interfacesJson;

    return networkDiagnostic;
}

bool MetricsCollectorBase::NetworkStatistics::isInterfaceInWhiteList(const std::string& iface) const {
    auto found = std::find_if(interfacesWhiteList_.cbegin(), interfacesWhiteList_.cend(),
                              [&iface](const std::string& arg) { return iface.find(arg, 0) == 0; });
    return found != interfacesWhiteList_.cend();
}

std::string MetricsCollectorBase::NetworkStatistics::getDefaultInterfaceName(std::istream& routingTable) {
    if (routingTable.fail()) {
        YIO_LOG_ERROR_EVENT("MetricsCollector.NetworkStatistics.RoutingTableStreamClosed", "Cannot collect NetworkStatistics - default gateway. Input stream closed")
    }
    std::string line;
    // find default gateway
    while (std::getline(routingTable, line)) {
        std::vector<std::string> fields;
        boost::split(fields, line, boost::is_any_of("\t "), boost::token_compress_on);
        if (fields.size() < ROUTING_TABLE_MIN_FIELDS_COUNT) {
            continue;
        }

        // if gateway is 0.0.0.0 (or 00000000 without formatting) return interface name
        if (fields[2] == DEFAULT_GATEWAY) {
            return fields[0];
        }
    }
    YIO_LOG_ERROR_EVENT("MetricsCollector.NetworkStatistics.DefaultGatewayNotFound", "Cannot collect NetworkStatistics - default gateway. Default gateway not found")
    return "";
}

std::vector<MetricsCollectorBase::NetworkStatistics::InterfaceInfo>
MetricsCollectorBase::NetworkStatistics::getInterfacesInfo(std::istream& ifacesStats) {
    if (ifacesStats.fail()) {
        YIO_LOG_ERROR_EVENT("MetricsCollector.NetworkStatistics.IfacesStreamClosed", "Cannot collect NetworkStatistics - interfaces info. Input stream closed")
        return {};
    }

    std::vector<InterfaceInfo> interfacesInfo;
    std::string line, headerLine;
    std::vector<std::string> headers, receiveHeaders, transmitHeaders;
    // skip first line (Inter-| Receive |  Transmit)
    std::getline(ifacesStats, line);
    std::getline(ifacesStats, headerLine);

    // read headers
    boost::split(headers, headerLine, boost::is_any_of("|"));
    if (headers.size() < IFACES_STATS_HEADERS_COUNT) {
        YIO_LOG_ERROR_EVENT("MetricsCollector.NetworkStatistics.InvalidIfaceHeadersSize", "Cannot collect NetworkStatistics - interfaces info. Incorrect headers format: " << headerLine)
        return {};
    }

    boost::trim_if(headers[1], boost::is_any_of("\t "));
    boost::trim_if(headers[2], boost::is_any_of("\t "));

    if (headers[1].empty() || headers[2].empty()) {
        YIO_LOG_ERROR_EVENT("MetricsCollector.NetworkStatistics.InvalidIfaceHeaders", "Cannot collect NetworkStatistics - interfaces info. Incorrect headers format: " << headerLine)
        return {};
    }

    boost::split(receiveHeaders, headers[1], boost::is_any_of("\t "), boost::token_compress_on);
    boost::split(transmitHeaders, headers[2], boost::is_any_of("\t "), boost::token_compress_on);

    while (std::getline(ifacesStats, line)) {
        std::istringstream ifaceData(line);
        InterfaceInfo info;
        // remove ':' at the end of a name
        ifaceData >> info.name;
        boost::trim_right_if(info.name, boost::is_any_of(":"));

        // collect received values
        for (const auto& statName : receiveHeaders) {
            ifaceData >> info.receiveStatistics[statName];
        }

        // collect transmitted values
        for (const auto& statName : transmitHeaders) {
            ifaceData >> info.transmitStatistics[statName];
        }
        interfacesInfo.push_back(info);
    }
    return interfacesInfo;
}

std::string MetricsCollectorBase::NetworkStatistics::getOperstateFilename(const std::string& ifaceName) {
    std::ostringstream filename;
    filename << INTERFACES_DIRECTORY << '/' << ifaceName << '/' << OPERSTATE_FILENAME;
    return filename.str();
}

std::string MetricsCollectorBase::NetworkStatistics::getOperstate(std::istream& operstate) {
    if (operstate.fail()) {
        YIO_LOG_ERROR_EVENT("MetricsCollector.NetworkStatistics.OperstateStreamClosed", "Cannot collect NetworkStatistics - get operstate. Input stream closed")
        return "";
    }
    // get interface state
    std::string state;
    operstate >> state;
    return state;
}

bool MetricsCollectorBase::shouldLogError(const std::string& message) const {
    const auto now = std::chrono::steady_clock::now();

    auto it = lastErrorTimes_.find(message);
    if (it == lastErrorTimes_.end()) {
        if (lastErrorTimes_.size() > MAX_LAST_ERROR_TIMES) {
            YIO_LOG_DEBUG("Error message was throttled (overflow): " << message);
            return false;
        }

        lastErrorTimes_.emplace(message, now);
        return true;
    }

    if (now - it->second > LOG_ERROR_INTERVAL) {
        it->second = now;
        return true;
    }

    YIO_LOG_DEBUG("Error message was throttled: " << message);
    return false;
}

const char* MetricsCollectorBase::matchProcWithService(const std::string& cmdline, const std::unordered_set<std::string>& services) {
    /*
     * From man 5 proc:
     * The command-line arguments appear in this file as a set of strings
     * separated by null bytes ('\0'), with a further null byte
     * after the last string.
     */
    const auto proc_argv = quasar::split(cmdline, std::string("\0", 1));
    if (proc_argv.empty())
        return nullptr;

    const char* service_name = nullptr;

    // First we check if binary name in argv[0] is matching any of the services
    const auto service = services.find(quasar::split(proc_argv[0], "/").back());
    if (service != services.end()) {
        service_name = service->c_str();
    }

    // Next we check if any of the arguments is equal to "--service_name"
    if (service_name == nullptr) {
        for (auto argv_i = std::next(proc_argv.begin()); argv_i != proc_argv.end(); ++argv_i) {
            if (argv_i->rfind("--", 0) != std::string::npos) {
                const auto it = services.find(std::string(argv_i->begin() + 2, argv_i->end()));
                if (it != services.end()) {
                    service_name = it->c_str();
                    break;
                }
            }
        }
    }

    return service_name;
}
