#pragma once

#include <util/str_stl.h> // DO NOT MOVE: workaround `util/str_stl.h` include-order dependency

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/metrics_collector/i_metrics_collector.h>
#include <yandex_io/libs/metrics_collector/numeric_stat.h>
#include <yandex_io/libs/metrics_collector/consumers/stat/aggregate_stat_metric_consumer.h>
#include <yandex_io/libs/threading/periodic_executor.h>
#include <yandex_io/metrica/monitor/metrics_collector/ping_manager/ping_manager.h>
#include <yandex_io/sdk/private/device_context.h>

#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/split.hpp>

#include <chrono>
#include <cstring>
#include <istream>
#include <limits>
#include <map>
#include <mutex>
#include <optional>
#include <unordered_map>
#include <unordered_set>

#include <sys/sysinfo.h>

namespace quasar {

    class MetricsCollectorBase
        : public IMetricsCollector {
    public:
        /**
         * Thermal nodes, which will be read in collectBaseMetrics.
         * Key is filename where temperature lies.
         * Value is pair of:
         * 1. Name of temperature stat which will be used in metrics.
         * 2. Measure units in Celsius degrees (e.g. value 0.001 means that measure value is millidegrees).
         */
        using ThermalNodes = std::map<std::string, std::pair<std::string, double>>;

        MetricsCollectorBase(std::shared_ptr<YandexIO::IDevice> device, const std::shared_ptr<ipc::IIpcFactory>& ipcFactory,
                             std::shared_ptr<PingManager> pingManager, ThermalNodes thermalNodes = ThermalNodes());
        virtual ~MetricsCollectorBase();

        Json::Value getMetrics() override;
        Json::Value getNetworkMetrics() override;

    public:
        static const char* matchProcWithService(const std::string& procPath, const std::unordered_set<std::string>& services);

    protected:
        void collectSystemMetrics(); // thread-safe

        void collectBaseMetrics(IMetricConsumer& metrics);             // Synchronized
        virtual void collectPlatformMetrics(IMetricConsumer& metrics); // Synchronized

        void collectCPUStats(IMetricConsumer& metrics, double timeElapsedMs);
        void collectMemStats(IMetricConsumer& metrics);
        void collectServicesInfo(IMetricConsumer& metrics, double timeElapsedMs);
        void collectThermalStats(IMetricConsumer& metrics);
        void collectUptime(IMetricConsumer& metrics);
        void collectLoadAverage(IMetricConsumer& metrics);

    private:
        Json::Value getNetworkMetricsLocked();

        void setConfig(const Json::Value& config) override;
        void setNetworkStatus(const proto::NetworkStatus& status) override;

        bool shouldLogError(const std::string& message) const;

    protected:
        struct CPUStats {
            int64_t idleMs = 0;
            int64_t userMs = 0;
            int64_t systemMs = 0;
            int64_t niceMs = 0;
            int64_t ioWaitMs = 0;
        };

        struct MemInfo {
            int64_t totalBytes = 0;
            int64_t freeBytes = 0;
            int64_t cachedBytes = 0;
            int64_t totalSwapBytes = 0;
            int64_t freeSwapBytes = 0;
            int64_t vmallocUsedBytes = 0;
            int64_t slabBytes = 0;
            int64_t mappedBytes = 0;
            int64_t majorPageFaults = 0;
        };

        struct PartitionInfo {
            std::string path;
            int64_t totalBytes = 0;
            int64_t freeBytes = 0;
        };

        struct QuasarServicesInfo {
            struct ServiceInfo {
                int64_t majorPageFaults = 0;
                int64_t utimeMs = 0;
                int64_t stimeMs = 0;
                int64_t runningThreads = -1;
                int64_t openFdCount = -1;
                int64_t vszKbytes = 0;
                int64_t rssKbytes = 0;
            };

            // Refer to fs/proc/array.c
            enum {
                MAJOR_FAULTS_IDX = 11,
                UTIME_IDX = 13,
                STIME_IDX = 14,
                NUM_THREADS_IDX = 19,
                VMSIZE_BYTES = 22,
                RSS_PAGES = 23,
                LAST_EXPECTED // Last expected value in enum. Used to verify size of read data
            };

            mutable std::unordered_map<const char*, ServiceInfo> servicesInfo_;

            bool empty() const {
                return servicesInfo_.empty();
            }

            const ServiceInfo& get(const char* serviceName) const {
                return servicesInfo_[serviceName];
            }

            void store(const char* serviceName, const ServiceInfo& info)
            {
                servicesInfo_[serviceName] = info;
            }

            ServiceInfo collectServiceInfo(const MetricsCollectorBase& collector, const std::string& proc_dir_path, int pageSizeKb) const;
        };

        struct NetworkStatistics {
            struct InterfaceInfo {
                std::string name;
                std::string operstate;
                std::unordered_map<std::string, int64_t> receiveStatistics;
                std::unordered_map<std::string, int64_t> transmitStatistics;
            };

            const static std::string IFACES_STATS_FILENAME;
            const static std::string ROUTING_TABLE_FILENAME;
            const static std::string INTERFACES_DIRECTORY;
            const static std::string OPERSTATE_FILENAME;
            const static std::string DEFAULT_GATEWAY;
            const static int ROUTING_TABLE_MIN_FIELDS_COUNT;
            const static int IFACES_STATS_HEADERS_COUNT;

            std::vector<std::string> interfacesWhiteList_;
            std::vector<std::string> dataWhiteList_;
            std::string defaultIfaceName_;
            std::vector<InterfaceInfo> interfaces_;

            NetworkStatistics();

            virtual void collect(const MetricsCollectorBase& collector);

            Json::Value toJson() const;

            bool isInterfaceInWhiteList(const std::string& iface) const;

            static std::string getDefaultInterfaceName(std::istream& routingTable);

            static std::vector<InterfaceInfo> getInterfacesInfo(std::istream& ifacesStats);

            static std::string getOperstateFilename(const std::string& ifaceName);

            static std::string getOperstate(std::istream& operstate);

            virtual ~NetworkStatistics() = default;
        };

        CPUStats getCPUStats() const;
        MemInfo getMemInfo() const;
        QuasarServicesInfo getServicesInfo() const;

        std::vector<PartitionInfo> getPartitionsInfo() const;

        std::shared_ptr<YandexIO::IDevice> device_;
        std::shared_ptr<PingManager> pingManager_;
        ThermalNodes thermalNodes_;

        struct ThermalStat {
            std::string thermalNodeName;
            double temperature;
        };
        std::vector<ThermalStat> getThermalNodesStats();

        std::unique_ptr<PeriodicExecutor> executor_;

        const std::vector<std::string> disksList_;
        uint32_t pageSizeKb_;
        std::vector<std::string> collectedProperties_;
        std::map<std::string, std::string> collectedPropertiesValues_;

        mutable std::mutex mutex_;
        AggregateStatMetricConsumer aggregateStatMetrics_;
        AggregateStatMetricConsumer platformAggregateStatMetrics_;
        std::map<std::string, std::vector<std::string>> platformCategoricalMetrics_;

        std::map<std::string, uint64_t> networkStatusesUsageMs_;

        YandexIO::DeviceContext deviceContext_;

        CPUStats lastCPUStats_;
        QuasarServicesInfo lastServicesInfo_;
        std::unordered_set<std::string> services_;
        using SteadyClock = std::chrono::steady_clock;
        std::optional<SteadyClock::time_point> lastMonotonicTimePoint_;
        int coreCount_;
        static long uptime()
        {
            struct sysinfo info;
            sysinfo(&info);

            return info.uptime;
        }

        void configurePropertiesReport(const Json::Value& propertiesReportConfig);

        std::chrono::steady_clock::time_point lastNetworkCheck_;
        proto::NetworkStatus::Status currentNetworkStatus_ = proto::NetworkStatus::NOT_CONNECTED;
        proto::ConnectionType currentConnectionType_ = proto::CONNECTION_TYPE_UNKNOWN;

        mutable std::unordered_map<std::string, std::chrono::steady_clock::time_point> lastErrorTimes_;
    };

} // namespace quasar
