#include "ping_manager.h"

#include <yandex_io/libs/logging/logging.h>

#include <unordered_set>

YIO_DEFINE_LOG_MODULE("ping_manager");

using namespace quasar;

namespace {
    const Pinger::Duration PING_DEFAULT_INTERVAL = std::chrono::seconds(10);
    const Pinger::Duration PING_DEFAULT_TIMEOUT = std::chrono::seconds(10);
    const Pinger::SocketType PING_DEFAULT_SOCKET_TYPE = Pinger::SocketType::RAW;
    const std::chrono::seconds GATEWAY_RELOAD_INTERVAL = std::chrono::seconds(15);
    const std::chrono::seconds PING_STATISTICS_WINDOW_SIZE = std::chrono::seconds(60);
}

PingManager::PingManager(OnEvent onEvent)
    : onEvent_(std::move(onEvent))
{
}

PingManager::~PingManager() {
    worker_.destroy();
    pinger_.reset();
    gatewayMonitor_.reset();
}

void PingManager::start(const Json::Value& pingerConfig) {
    configurePinger(pingerConfig);
}

void PingManager::stop() {
    worker_.add([this]() {
        std::lock_guard<std::mutex> lock(mutex_);
        for (auto it = pings_.begin(); it != pings_.end();) {
            pinger_->stopPing(it->second);
            it = pings_.erase(it);
        }
        pingGateway_ = false;
    });
}

void PingManager::configurePinger(const Json::Value& cfg) {
    std::unordered_set<Ping, Ping::Hash> pings;
    bool pingGateway = false;

    std::lock_guard<std::mutex> lock(mutex_);

    auto pingsConfig = cfg["pings"];
    if (!pingsConfig.isNull()) {
        for (auto it = pingsConfig.begin(); it != pingsConfig.end(); ++it) {
            Ping ping;

            try {
                ping.interval = std::chrono::milliseconds(tryGetUInt64(*it, "intervalMs", 0));
                ping.timeout = std::chrono::milliseconds(tryGetUInt64(*it, "timeoutMs", 0));

                const bool gateway = tryGetBool(*it, "gateway", false);

                if (pingGateway && gateway) {
                    // multiple gateways in config, skip
                    continue;
                }

                if (gateway) {
                    pingGateway = true;
                    ping.host = gatewayPing_.host;
                    gatewayPing_ = ping;

                } else {
                    ping.host = tryGetString(*it, "host", "");
                }

            } catch (const Json::LogicError& e) {
                YIO_LOG_ERROR_EVENT("PingManager.Pings.InvalidConfig", "Bad pings config: " << e.what());
                continue;
            }

            pings.insert(ping);
        }
    }

    if (!pings.empty() && !pinger_) {
        pinger_ = createPinger(cfg);
    }

    pingGateway_ = pingGateway;

    if (pingGateway && !gatewayMonitor_) {
        gatewayMonitor_ = createGatewayMonitor();
    }

    for (const auto& newPing : pings) {
        if (pings_.count(newPing) == 0) {
            pings_[newPing] = pinger_->startPing(newPing.host, newPing.interval, newPing.timeout);
        }
    }

    for (auto it = pings_.begin(); it != pings_.end();) {
        if (pings.count(it->first) == 0) {
            pinger_->stopPing(it->second);
            it = pings_.erase(it);
        } else {
            ++it;
        }
    }
}

std::shared_ptr<Pinger> PingManager::createPinger(const Json::Value& config) {
    std::string socketTypeStr = tryGetString(config, "socketType", "");
    Pinger::SocketType socketType;

    if (socketTypeStr == "raw") {
        socketType = Pinger::SocketType::RAW;
    } else if (socketTypeStr == "dgram") {
        socketType = Pinger::SocketType::DGRAM;
    } else {
        socketType = PING_DEFAULT_SOCKET_TYPE;
    }

    return Pinger::create(PING_DEFAULT_INTERVAL, PING_DEFAULT_TIMEOUT, this, &worker_, socketType);
}

std::shared_ptr<GatewayMonitor> PingManager::createGatewayMonitor() {
    return GatewayMonitor::create(GATEWAY_RELOAD_INTERVAL, this, &worker_);
}

void PingManager::onEvent(const Pinger::Event& ev) {
    OnEvent onEventCallback;
    {
        std::lock_guard<std::mutex> lock(mutex_);

        if (ev.type == Pinger::EventType::PACKET_SENT) {
            const auto ins = pingRequests_.insert(std::make_pair(
                ev.sequenceNumber,
                PingRequest{ev.pingId, ev.type, ev.time}));

            Y_VERIFY(ins.second);
            if (!ins.second) {
                YIO_LOG_ERROR_EVENT("PingManager.Pings.PacketAlreadySent",
                                    "Packet with same seq already sent");
                return;
            }

            pingRequestsOrder_.push(ev.sequenceNumber);

            auto it = windowPingStats_.find(ev.pingId);

            if (it == windowPingStats_.end()) {
                auto& st = windowPingStats_[ev.pingId];
                st.host = ev.host;
                st.isGateway = (ev.host == gatewayPing_.host);
                ++st.packetsSent;

            } else {
                ++it->second.packetsSent;
            }

        } else {
            auto it = pingRequests_.find(ev.sequenceNumber);

            if (it == pingRequests_.end()) {
                // skip
                return;
            }

            auto& request = it->second;
            auto& st = windowPingStats_[ev.pingId];

            if (ev.type == Pinger::EventType::PACKET_RECEIVED) {
                Y_VERIFY(request.status == Pinger::EventType::PACKET_SENT);

                request.status = Pinger::EventType::PACKET_RECEIVED;

                ++st.packetsReceived;

                st.pingDurations.push_back(
                    std::chrono::duration_cast<std::chrono::milliseconds>(
                        ev.elapsed)
                        .count());

                YIO_LOG_DEBUG("ping " << ev.host << ", time=" << st.pingDurations.back()
                                      << "ms");

            } else if (ev.type == Pinger::EventType::PACKET_RECEIVED_DUPLICATE) {
                Y_VERIFY(request.status == Pinger::EventType::PACKET_RECEIVED ||
                         request.status == Pinger::EventType::PACKET_RECEIVED_DUPLICATE);

                if (request.status == Pinger::EventType::PACKET_RECEIVED) {
                    ++st.packetsDuplicate;
                }

                request.status = Pinger::EventType::PACKET_RECEIVED_DUPLICATE;

            } else if (ev.type == Pinger::EventType::PACKET_LOST) {
                Y_VERIFY(request.status == Pinger::EventType::PACKET_SENT);

                request.status = Pinger::EventType::PACKET_LOST;

                ++st.packetsLost;

                YIO_LOG_DEBUG("ping " << ev.host << ", packet_lost");

            } else {
                // never happens
            }
        }
        stripPingRequestsNoLock();
        onEventCallback = onEvent_;
    }

    if (onEventCallback) {
        onEventCallback(ev);
    }
}

void PingManager::stripPingRequestsNoLock() {
    const auto windowStart = Pinger::Clock::now() - PING_STATISTICS_WINDOW_SIZE;

    while (!pingRequestsOrder_.empty()) {
        const uint16_t sequenceNumber = pingRequestsOrder_.front();

        const auto requestIt = pingRequests_.find(sequenceNumber);

        Y_VERIFY(requestIt != pingRequests_.end());
        if (requestIt == pingRequests_.end()) {
            // Should never happen, skip
            YIO_LOG_ERROR_EVENT("PingManager.Pings.PingRequestNotFound", "Ping request with sequenceNumber " << sequenceNumber << " not found");
            pingRequestsOrder_.pop();
            continue;
        }

        const auto& request = requestIt->second;

        if (request.sendTime >= windowStart) {
            break;
        }

        auto statsIt = windowPingStats_.find(request.pingId);

        Y_VERIFY(statsIt != windowPingStats_.end());
        if (statsIt == windowPingStats_.end()) {
            // Should never happen, skip
            YIO_LOG_ERROR_EVENT("PingManager.Pings.NoStatForRequestedPing", "No stats for ping " << request.pingId);
            pingRequestsOrder_.pop();
            pingRequests_.erase(requestIt);
            continue;
        }

        auto& st = statsIt->second;

        switch (request.status) {
            case Pinger::EventType::PACKET_RECEIVED_DUPLICATE:
                --st.packetsDuplicate;
                // fallthrough

            case Pinger::EventType::PACKET_RECEIVED:
                --st.packetsReceived;
                st.pingDurations.pop_front();
                // fallthrough

            case Pinger::EventType::PACKET_SENT:
                --st.packetsSent;
                break;

            case Pinger::EventType::PACKET_LOST:
                --st.packetsLost;
                --st.packetsSent;
                break;

            default:
                // Should never happen
                break;
        }

        if (st.packetsSent == 0) {
            windowPingStats_.erase(statsIt);
        }

        pingRequestsOrder_.pop();
        pingRequests_.erase(requestIt);
    }
}

void PingManager::onGatewayChanged(const std::string& /* oldIp */, const std::string& newIp) {
    std::lock_guard<std::mutex> lock(mutex_);

    if (pingGateway_) {
        auto it = pings_.find(gatewayPing_);

        Y_VERIFY(it != pings_.end());

        if (it == pings_.end()) {
            // Should never happen
            YIO_LOG_ERROR_EVENT("PingManager.GatewayMonitor.PingNotFound", "Gateway ping not found");

        } else {
            pinger_->stopPing(it->second);
            pings_.erase(it);
        }

        gatewayPing_.host = newIp;

        it = pings_.find(gatewayPing_);

        Y_VERIFY(it == pings_.end());

        if (it == pings_.end()) {
            pings_[gatewayPing_] = pinger_->startPing(gatewayPing_.host, gatewayPing_.interval, gatewayPing_.timeout);

        } else {
            // Should never happen
            YIO_LOG_ERROR_EVENT("PingManager.GatewayMonitor.PingAlreadyExists", "New gateway ping already exists");
        }

    } else {
        gatewayPing_.host = newIp;
    }
}

void PingManager::reloadGateway() {
    std::lock_guard<std::mutex> lock(mutex_);
    if (gatewayMonitor_) {
        gatewayMonitor_->reloadNow();
    }
}

std::unordered_map<uint64_t, PingManager::PingStatistics> PingManager::getRecentPingsStatistics() {
    std::lock_guard<std::mutex> lock(mutex_);
    stripPingRequestsNoLock();
    return windowPingStats_;
}
