#pragma once

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/ping/pinger.h>
#include <yandex_io/libs/ping/gateway_monitor.h>

namespace quasar {
    class PingManager
        : public Pinger::Listener,
          public GatewayMonitor::Listener {
    public:
        using OnEvent = std::function<void(const Pinger::Event&)>;

        struct PingRequest {
            uint64_t pingId;
            Pinger::EventType status;
            Pinger::TimePoint sendTime;
        };

        struct Ping {
            std::string host;
            Pinger::Duration interval;
            Pinger::Duration timeout;

            bool operator==(const Ping& oth) const {
                return host == oth.host && interval == oth.interval && timeout == oth.timeout;
            }

            struct Hash {
                size_t operator()(const Ping& ping) const {
                    return std::hash<std::string>()(ping.host);
                }
            };
        };

        struct PingStatistics {
            std::string host;
            bool isGateway;

            uint64_t packetsSent = 0;
            uint64_t packetsReceived = 0;
            uint64_t packetsLost = 0;
            uint64_t packetsDuplicate = 0;

            std::deque<int64_t> pingDurations;
        };

        PingManager(OnEvent onEvent = nullptr);
        ~PingManager();
        void start(const Json::Value& pingerConfig);
        void stop();
        void onEvent(const Pinger::Event& event) override;
        void onGatewayChanged(const std::string& oldIp, const std::string& newIp) override;
        void reloadGateway();
        std::unordered_map<uint64_t, PingStatistics> getRecentPingsStatistics();

    private:
        void configurePinger(const Json::Value& cfg);
        std::shared_ptr<Pinger> createPinger(const Json::Value& config);
        std::shared_ptr<GatewayMonitor> createGatewayMonitor();
        void stripPingRequestsNoLock();

        std::mutex mutex_;
        Ping gatewayPing_;
        bool pingGateway_;
        std::shared_ptr<Pinger> pinger_;
        std::shared_ptr<GatewayMonitor> gatewayMonitor_;
        std::unordered_map<Ping, uint64_t, Ping::Hash> pings_;
        NamedCallbackQueue worker_{"PingManager"};
        std::unordered_map<uint16_t, PingRequest> pingRequests_;
        std::queue<uint16_t> pingRequestsOrder_;
        std::unordered_map<uint64_t, PingStatistics> windowPingStats_;
        OnEvent onEvent_;
    };
} // namespace quasar
