#include "monitor_endpoint.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/filesystem/directory_iterator.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/pstore_collector/pstore_collector.h>
#include <yandex_io/libs/net/wifi_info.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <cstdlib>

#include <sys/sysinfo.h>

YIO_DEFINE_LOG_MODULE("monitor");

using namespace quasar;

MonitorEndpoint::MonitorEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::unique_ptr<IMetricsCollector> collector)
    : device_(std::move(device))
    , metricsCollector_(std::move(collector))
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
{
    server_->setMessageHandler([this](const auto& message, auto& connection) {
        handleQuasarMessage(message, &connection);
    });
    server_->listenService();

    const auto serviceConfig = device_->configuration()->getServiceConfig("monitord");
    const int dumpPeriodSec = quasar::tryGetInt(serviceConfig, "metricsDumperPeriodSec", 0);
    auto diskUsageConfig = quasar::tryGetJson(serviceConfig, "diskUsage", Json::Value::null);
    if (diskUsageConfig.isArray()) {
        parseDiskUsageConfig(diskUsageConfig);
    }
    std::chrono::milliseconds dumpPeriod = std::chrono::minutes(5);
    if (dumpPeriodSec > 0) {
        dumpPeriod = std::chrono::seconds(dumpPeriodSec);
    } else {
        YIO_LOG_INFO("Parameter metricsDumperPeriodSec is not in settings, using default parameter");
    }

    dumpMetricsExecutor_ = std::make_unique<PeriodicExecutor>(
        std::bind(&MonitorEndpoint::dumpMetrics, this),
        dumpPeriod);

    resetDiskUsageCollector();

    const int wifiPeriodSec = quasar::tryGetInt(serviceConfig, "wifiStatsPeriodSec", 0);
    if (wifiPeriodSec > 0) {
        wifiStatsPeriod_ = std::chrono::seconds(wifiPeriodSec);
        sendWifiStats_ = true;
        resetWifiStatsCollector();
    }

    collectPstore(serviceConfig["pstorePath"].asString());
}

void MonitorEndpoint::parseDiskUsageConfig(const Json::Value& config) {
    pathsToMonitor_.clear();
    if (!config.isArray()) {
        return;
    }
    for (Json::ArrayIndex index = 0; index < config.size(); ++index) {
        const auto& entry = config[index];
        DiskUsagePath element;
        if (!entry.isMember("root")) {
            YIO_LOG_ERROR_EVENT("MonitorEndpoint.DiskUsage.InvalidDiskEntryJson", "root element not found in config");
            continue;
        }
        element.root = entry["root"].asString();
        element.maxDepth = tryGetInt(entry, "maxDepth", 4);
        YIO_LOG_INFO("Monitor " << element.root << " with maxDepth = " << element.maxDepth);
        pathsToMonitor_.push_back(std::move(element));
    }
}

void MonitorEndpoint::resetDiskUsageCollector() {
    if (!collectDiskUsageInfo_.load() || pathsToMonitor_.size() == 0) {
        YIO_LOG_DEBUG("Disk usage info will not be collecting");
        diskUsageExecutor_.reset();
    } else {
        YIO_LOG_DEBUG("Will run disk usage collecting with a period of " << diskUsageMonitorPeriod_.count() << " milliseconds");
        diskUsageExecutor_ = std::make_unique<PeriodicExecutor>([this]() {
            collectDiskUsageStats();
        }, diskUsageMonitorPeriod_, PeriodicExecutor::PeriodicType::SLEEP_FIRST);
    }
}

void MonitorEndpoint::resetWifiStatsCollector() {
    if (!sendWifiStats_ || wifiStatsPeriod_ == std::chrono::seconds(0)) {
        YIO_LOG_DEBUG("Wifi satistics will not be collecting");
        wifiStatsExecutor_.reset();
    } else {
        YIO_LOG_INFO("Wifi statistics interval is " << wifiStatsPeriod_.count() << " seconds");
        wifiStatsExecutor_ = std::make_unique<PeriodicExecutor>(
            std::bind(&MonitorEndpoint::wifiStats, this),
            wifiStatsPeriod_);
    };
}

void MonitorEndpoint::collectDiskUsageStats() {
    YIO_LOG_INFO("Collecting disk usage info");

    for (const auto& path : pathsToMonitor_) {
        // Only logging info from this call is required. Ignore the result
        collectDirectoryDiskUsageInfo(path.root, path.maxDepth);
    }
}

int64_t MonitorEndpoint::collectDirectoryDiskUsageInfo(const std::string& path, int maxDepth) {
    struct stat pathStat;
    if (stat(path.c_str(), &pathStat) < 0) {
        YIO_LOG_ERROR_EVENT("MonitorEndpoint.DiskUsage.StatFailed", "Failed to access path " << path << ", errorCode = " << errno);
        return 0;
    }
    if (S_ISDIR(pathStat.st_mode)) {
        if (maxDepth <= 0) {
            return 0;
        }

        int64_t totalDirectorySize = 0;
        for (auto iter = DirectoryIterator(path); iter.isValid(); iter.next()) {
            auto subpath = iter.currentEntryName();
            if (subpath == "." || subpath == "..") {
                continue;
            }
            totalDirectorySize += collectDirectoryDiskUsageInfo(path + "/" + subpath, maxDepth - 1);
        }
        YIO_LOG_INFO("DiskUsageInfo: " << path << " " << totalDirectorySize);
        return totalDirectorySize;
    } else if (S_ISREG(pathStat.st_mode)) {
        size_t totalSize = pathStat.st_blksize * pathStat.st_blocks;
        YIO_LOG_INFO("DiskUsageInfo: " << path << " " << totalSize);
        return totalSize;
    } else {
        return 0;
    }
}

bool MonitorEndpoint::applyWifiStatsSettings(const Json::Value& cfg) {
    if (const auto newWifiStatsPeriod = tryGetInt(cfg, "periodSec", 0); newWifiStatsPeriod > 0) {
        {
            net::WifiInfoSettings newSettings;
            newSettings.allBss = tryGetBool(cfg, "allBss", false);
            newSettings.iface = tryGetBool(cfg, "iface", false);
            newSettings.tids = tryGetBool(cfg, "tids", false);
            newSettings.bothIE = tryGetBool(cfg, "bothIE", false);
            newSettings.longCpblt = tryGetBool(cfg, "longCpblt", false);
            newSettings.survey = tryGetBool(cfg, "survey", false);
            std::scoped_lock<std::mutex> lock(wifiStatsSettingsMutex_);
            wifiStatsSettings_ = newSettings;
        }
        const auto oldFlag = sendWifiStats_.exchange(true);
        const auto newValue = std::chrono::seconds(newWifiStatsPeriod);
        if (newValue != wifiStatsPeriod_ || !oldFlag) {
            wifiStatsPeriod_ = newValue;
            resetWifiStatsCollector();
        }
        return true;
    }
    return false;
}

void MonitorEndpoint::updateConfig(const std::string& config) {
    auto configOptional = tryParseJson(config);
    if (configOptional.has_value()) {
        auto& value = configOptional.value()["system_config"];
        bool changed = false;
        if (auto diskUsageMonitorPeriod = std::chrono::seconds(
                tryGetInt(value, "diskUsageMonitorPeriodSeconds", DEFAULT_DISK_MONITOR_PERIOD_SECONDS));
            diskUsageMonitorPeriod != diskUsageMonitorPeriod_)
        {
            diskUsageMonitorPeriod_ = diskUsageMonitorPeriod;
            changed = true;
        }

        if (!value.isMember("wifi_stats") || !applyWifiStatsSettings(value["wifi_stats"])) {
            sendWifiStats_ = false;
            resetWifiStatsCollector();
        }

        if (auto collectDiskUsageInfo = tryGetBool(value, "collectDiskUsageInfo", false); collectDiskUsageInfo != collectDiskUsageInfo_) {
            collectDiskUsageInfo_ = collectDiskUsageInfo;
            changed = true;
        }
        if (changed) {
            resetDiskUsageCollector();
        }

        metricsCollector_->setConfig(*configOptional);
    }
}

void MonitorEndpoint::handleQuasarMessage(const ipc::SharedMessage& message, ipc::IServer::IClientConnection* /*connection*/)
{
    if (message->has_metrica_message()) {
        const auto& metricaMessage = message->metrica_message();

        if (metricaMessage.has_config()) {
            updateConfig(metricaMessage.config());
        }

        if (metricaMessage.has_network_status()) {
            metricsCollector_->setNetworkStatus(metricaMessage.network_status());
        }
    }
}

void MonitorEndpoint::dumpMetrics()
{
    auto metricsJson = metricsCollector_->getMetrics();
    device_->telemetry()->reportEvent("systemMetrics", jsonToString(metricsJson, true));
    if (metricsJson.isMember("pings")) {
        const auto& pings = metricsJson["pings"];
        if (!pings.isNull() && pings.isObject()) {
            device_->telemetry()->reportEvent("pings", jsonToString(pings, true));
        }
    }
    YIO_LOG_TRACE("Reporting system metrics: " << metricsJson);
}

net::WifiInfoSettings MonitorEndpoint::getWifiStatsSettings() {
    std::scoped_lock<std::mutex> lock(wifiStatsSettingsMutex_);
    return wifiStatsSettings_;
}

void MonitorEndpoint::wifiStats()
{
    try {
        const auto settings = getWifiStatsSettings();
        YIO_LOG_INFO("WifiInfo will collect " << (settings.allBss ? "all bss" : "only my bss"));
        device_->telemetry()->reportEvent("wifiStats", jsonToString(net::getDetailedWifiInfo("wlan0", settings)));
    } catch (const std::runtime_error& e) {
        YIO_LOG_WARN("Cannot get wifi information: " << e.what());
    }
}

void MonitorEndpoint::collectPstore(const std::string& pstorePath)
{
    if (pstorePath.empty()) {
        return;
    }

    pstoreCollectorExecutor_ = std::make_unique<PeriodicExecutor>(
        [this, pstorePath]() {
            try {
                PstoreCollector(pstorePath, device_).collect();
            } catch (const std::runtime_error& e) {
                YIO_LOG_ERROR_EVENT("MonitorEndpoint.Pstore.CollectFailed", "pstore collector failed: " << e.what());
            }
        },
        std::chrono::milliseconds::zero(),
        PeriodicExecutor::PeriodicType::ONE_SHOT);
}

MonitorEndpoint::~MonitorEndpoint()
{
    dumpMetricsExecutor_.reset(nullptr);
    diskUsageExecutor_.reset(nullptr);
}

const std::string MonitorEndpoint::SERVICE_NAME = "monitord";
