#include "appmetrica_batcher.h"

#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/appmetrica/startup_client.h>
#include <yandex_io/libs/appmetrica/metrica_metadata.h>
#include <yandex_io/libs/appmetrica/app_metrica_utils.h>
#include <yandex_io/libs/metrica/db/utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <future>

using namespace quasar;
using namespace quasar::proto;
using namespace YandexIO;

namespace {
    constexpr int MAX_REPORT_SIZE = 1024 * 200; // We don't send reports with size more than 200 Kb
    constexpr std::chrono::seconds MAX_REPORT_INTERVAL{300};

    ReportMessage::Session::Event convertDbEvent(const ITelemetryEventsDB::Event& src, unsigned idInBatch) {
        ReportMessage::Session::Event result;
        result.set_name(src.name());
        result.set_time(src.timestamp() - src.session_start_time());

        // Account deprecated
        if (src.has_account_type()) {
            result.mutable_account()->set_type(src.account_type());
        }
        if (src.has_account_id()) {
            result.mutable_account()->set_id(src.account_id());
        }

        if (src.type() == DatabaseMetricaEvent::NewEvent::ERROR) {
            result.set_type(ReportMessage::Session::Event::EVENT_PROTOBUF_ERROR);
        } else {
            result.set_type(ReportMessage::Session::Event::EVENT_CLIENT);
        }
        result.set_value(src.value());
        if (src.has_serial_number()) {
            result.set_number_in_session(src.serial_number());
        } else {
            // backward compatibility, in future should simply drop events without number
            result.set_number_in_session(idInBatch);
        }

        result.mutable_network_info()->set_connection_type(toMetricaEventConnectionType(src.connection_type()));
        return result;
    }

    // appmetricas backends allows keepalive connection to live about 100-200 seconds
    std::vector<std::chrono::seconds> makeIntervals() {
        std::vector<std::chrono::seconds> result(1, MAX_REPORT_INTERVAL);
        for (int i = 0; i < 3; ++i) {
            result.push_back(std::chrono::seconds(20));
        }
        return result;
    }

} // namespace

AppmetricaBatcher::IntervalsRing::IntervalsRing(std::vector<std::chrono::seconds> intervals)
    : ring(std::move(intervals))
    , curPos(0)
{
}

void AppmetricaBatcher::IntervalsRing::next() {
    if (++curPos >= ring.size()) {
        curPos = 0;
    }
}

void AppmetricaBatcher::IntervalsRing::reset() {
    curPos = 0;
}

std::chrono::seconds AppmetricaBatcher::IntervalsRing::get() const {
    return ring[curPos];
}

// batch

AppmetricaBatch::AppmetricaBatch() {
    sessionDesc_.set_session_type(ReportMessage::Session::SessionDesc::SESSION_FOREGROUND);
    sessionDesc_.set_locale("ru-RU");
    sessionDesc_.mutable_start_time()->set_time_zone(3 * 60 * 60); // moscow time
}

std::uint32_t AppmetricaBatch::size() const {
    return size_;
}

void AppmetricaBatch::reset() {
    batch_ = ReportMessage();
    completed_ = false;
    size_ = 0;
    if (unsentEvent_) {
        tryAdd(std::move(*unsentEvent_));
        unsentEvent_.reset();
    }
}

void AppmetricaBatch::complete() {
    if (!completed_) {
        auto sendTime = batch_.mutable_send_time();
        const auto now = std::chrono::system_clock::now();
        auto timestampInSec = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count();
        sendTime->set_timestamp(static_cast<uint64_t>(timestampInSec));
        sendTime->set_time_zone(batch_.sessions(0).session_desc().start_time().time_zone());
        applyEnv();
        completed_ = true;
    }
}

void AppmetricaBatch::applyEnv() {
    for (const auto& [name, value] : currentEnv_.value().environment_values()) {
        auto report_env = batch_.add_app_environment();
        report_env->set_name(name);
        report_env->set_value(value);
    }
}

bool AppmetricaBatch::tryAdd(ITelemetryEventsDB::Event srcEvent) {
    ReportMessage::Session::Event dstEvent = convertDbEvent(srcEvent, size_);
    const auto eventSerializedSize = dstEvent.SerializeAsString().size();
    if (eventSerializedSize > MAX_REPORT_SIZE) {
        YIO_LOG_WARN("Event '" << srcEvent.name() << "' dropped cos it's serialized size " << eventSerializedSize << " exceeded limit " << MAX_REPORT_SIZE);
        return true;
    }
    // event is ok, can be processed into batch

    const auto reportSerializedSize = batch_.SerializeAsString().size();
    if (reportSerializedSize + eventSerializedSize > MAX_REPORT_SIZE) {
        YIO_LOG_INFO("Cummulative serialized size of batch with new event exceeded limit. Delay event to next batch");
        complete();
        unsentEvent_ = std::move(srcEvent);
        return false;
    }

    ++size_;
    if (batch_.sessions().empty() || batch_.sessions().rbegin()->id() != srcEvent.session_id()) {
        auto session = batch_.add_sessions();
        session->set_id(srcEvent.session_id());

        auto sessionDesc = session->mutable_session_desc();
        *sessionDesc = sessionDesc_;

        sessionDesc->mutable_start_time()->set_timestamp(srcEvent.session_start_time());

        session->add_events()->Swap(&dstEvent);
    } else {
        batch_.mutable_sessions()->rbegin()->add_events()->Swap(&dstEvent);
    }

    return true;
}

bool AppmetricaBatch::append(ITelemetryEventsDB::Event event, ITelemetryEventsDB::Environment env) {
    if (currentEnv_.has_value() && !equalEnvs(currentEnv_.value(), env)) {
        if (size_ == 0) {
            currentEnv_ = std::move(env);
            return tryAdd(std::move(event));
        };
        YIO_LOG_INFO("Env changed! Left this event for next batch");
        complete();
        // store for next batch
        unsentEvent_ = std::move(event);
        currentEnv_ = std::move(env);
        return false;
    } else if (!currentEnv_.has_value()) {
        currentEnv_ = std::move(env);
    }
    return tryAdd(std::move(event));
}

bool AppmetricaBatch::hasUnsent() const {
    return unsentEvent_.has_value();
}

// batcher

AppmetricaBatcher::AppmetricaBatcher(std::shared_ptr<YandexIO::IDevice> device)
    : Batcher("appmeticaBatcher")
    , device_(std::move(device))
    , intervals_(makeIntervals())
    , sender_(device_, true)
{
    // init common sessions description
    const auto& serviceConfig = device_->configuration()->getServiceConfig("metricad");

    const std::string appMetricaStartupHost = serviceConfig["appMetricaStartupHost"].asString();
    const std::string appMetricaApiKey = serviceConfig["apiKey"].asString();
    const std::string metricaMetadataPath = serviceConfig.isMember("metricaMetadataPath")
                                                ? serviceConfig["metricaMetadataPath"].asString()
                                                : DEFAULT_METRICA_METADATA_PATH;

    const auto& commonConfig = device_->configuration()->getServiceConfig("common");
    const std::string deviceType = commonConfig["deviceType"].asString();
    const std::string metricaAppId = commonConfig["metricaAppId"].asString();

    auto metricaMetadata = MetricaMetadata::loadMetadata(metricaMetadataPath);

    metricaMetadata.deviceID = device_->deviceId();

    const auto appVersionName = device_->softwareVersion();
    auto appVersion = appVersionName;
    appVersion.erase(std::remove_if(appVersion.begin(), appVersion.end(),
                                    [](char c) { return !std::isdigit(c); }), appVersion.end());

    auto startupConfig = StartupConfiguration{
        .startTime = std::time(nullptr),
        .startupHost = appMetricaStartupHost,
        .apiKey = appMetricaApiKey,
        .deviceID = metricaMetadata.deviceID,
        .UUID = metricaMetadata.UUID,
        .model = deviceType,
        .appVersion = appVersion,
        .appVersionName = appVersionName,
        .metricaAppID = metricaAppId,
        .revision = device_->platformRevision(),
        .osVersion = StartupConfiguration::systemOsVersion(),
    };

    queue_->add([this, startupConfig = std::move(startupConfig)]() mutable {
        bootstrap(std::move(startupConfig));
    });
}

void AppmetricaBatcher::setTimezoneOffsetSec(int32_t offsetSec) {
    queue_->add([this, offsetSec]() {
        batch_.sessionDesc_.mutable_start_time()->set_time_zone(offsetSec);
    });
}

void AppmetricaBatcher::setLocation(double lat, double lon) {
    queue_->add([this, lat, lon]() {
        auto location = batch_.sessionDesc_.mutable_location();
        location->set_lat(lat);
        location->set_lon(lon);
    });
}

std::chrono::seconds AppmetricaBatcher::getMaxBatchCollectingTime() {
    return intervals_.get();
}

bool AppmetricaBatcher::sendBatch() {
    batch_.complete();
    return sender_.trySendReport(batch_.batch_, reportConfig_->reportRequestURLS());
}

void AppmetricaBatcher::ready() {
    ready_ = true;

    auto toStr = [](auto container) {
        std::string result;
        for (auto s : container) {
            result += ' ';
            result += s;
        }
        return result;
    };

    YIO_LOG_INFO("Ready! " << toStr(reportConfig_->reportHosts()));
    if (readyCallback_) {
        readyCallback_();
    }
}

/* startup methods */

void AppmetricaBatcher::setOnReady(std::function<void()> callback) {
    std::promise<void> promise;
    queue_->add([this, &promise, callback = std::move(callback)]() {
        if (ready_) {
            queue_->add([callback]() {
                callback();
            });
        } else {
            readyCallback_ = callback;
        }
        promise.set_value();
    });
    promise.get_future().get();
}

void AppmetricaBatcher::bootstrap(StartupConfiguration startupConfig) {
    auto getBootstrapConfig = [this](const StartupConfiguration& startupConfig) -> std::optional<ResponseData> {
        try {
            return StartupClient::tryGetBootstrapConfig(device_, startupConfig.startupRequestURL(startupConfig.startupHost));
        } catch (...) {
        }
        return std::nullopt;
    };

    auto bootstrapCfg = getBootstrapConfig(startupConfig);
    if (bootstrapCfg) {
        YIO_LOG_INFO("Bootstrap config received, gathering another part");
        queue_->add([this, startupConfig, bootstrapCfg = *bootstrapCfg]() {
            reportConfig_ = StartupClient::getReportConfigFromBootstrap(device_, bootstrapCfg, startupConfig);
            YIO_LOG_INFO("Report configuration collected");
            ready();
        });
    } else {
        YIO_LOG_INFO("Bootstrap attempt failed, delayed next attempt");
        queue_->addDelayed([this, startupConfig = std::move(startupConfig)]() {
            bootstrap(startupConfig);
        }, std::chrono::seconds(5)); // FIXME: increase to 30 after first try
    }
};
